#! /usr/bin/perl

##  Copyright 2010-2022 Carnegie Mellon University
##  See license information in LICENSE.txt.

##  Reads the NEWS file on stdin and writes the releases.xml file to stdout.
##  Requires at least two arguments: (1)the name of this project, (2)the
##  license name to include in the click-through when downloading the source
##  code.  An optional third argument specifies the number of releases that
##  should include download links.
##  ------------------------------------------------------------------------
##  @DISTRIBUTION_STATEMENT_BEGIN@
##  YAF 3.0.0
##
##  Copyright 2022 Carnegie Mellon University.
##
##  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
##  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
##  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
##  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
##  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
##  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
##  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
##  INFRINGEMENT.
##
##  Released under a GNU GPL 2.0-style license, please see license.txt or
##  contact permission@sei.cmu.edu for full terms.
##
##  [DISTRIBUTION STATEMENT A] This material has been approved for public
##  release and unlimited distribution.  Please see Copyright notice for
##  non-US Government use and distribution.
##
##  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
##  Trademark Office by Carnegie Mellon University.
##
##  This Software includes and/or makes use of Third-Party Software subject
##  to its own license.
##
##  DM22-0007
##  @DISTRIBUTION_STATEMENT_END@
##  ------------------------------------------------------------------------


use warnings;
use strict;

# number of releases to allow downloads for (unlimited)
my $relkeep = ~0;

my $project = $ARGV[0] or die "Expected package name as first argument\n";
my $license = $ARGV[1] or die "Expected license name as second argument\n";
if ($ARGV[2]) {
    # If provided, we'll only spit out <p:file> elements for this number
    # of releases.  We'll still show history for the others, but won't
    # provide download links for them.
    $relkeep = int($ARGV[2]) or die "Expected integer as third argument: $!\n";
}

print <<HEAD;
<?xml version="1.0"?>
<p:project xmlns:p="http://netsa.cert.org/xml/project/1.0"
           xmlns="http://www.w3.org/1999/xhtml"
           xmlns:xi="http://www.w3.org/2001/XInclude">
HEAD

# slurp in all of the standard input
my $content;
{
    local $/ = undef;
    $content = <STDIN>;
}

my $relcount = 0;

# This regexp is pretty liberal, so as to be able to grok most NEWS formats.
while ($content =~ /^Version (\d[^:]*?):?\s+\(?([^\n]+?)\)?\s*\n\s*=+\s*((?:.(?!Version))+)/msg)
{
    my ($ver, $date, $notes) = ($1, $2, $3);

    if ($notes =~ /SPONSOR ONLY RELEASE/) {
        next;
    }

    ++$relcount;

    print <<RELHEAD1;
  <p:release>
    <p:version>$ver</p:version>
    <p:date>$date</p:date>
RELHEAD1

    if ($relcount <= $relkeep) {
        print <<RELHEAD2;
    <p:file href="../releases/$project-$1.tar.gz" license="$license"/>
RELHEAD2
    }

    print <<RELHEAD3;
    <p:changelog>
      <ul>
RELHEAD3

    # html escape the notes
    $notes =~ s/&/&amp;/g;
    $notes =~ s/</&lt;/g;
    $notes =~ s/>/&gt;/g;

    # First, see if items are delimited by \n\n
    if ($notes =~ m@(.+?)\n\n+?@) {
        while ($notes =~ m@(.+?)\n\n+?@msg) {
            print " " x 8, "<li>$1</li>\n";
        }
        # The last item will be skipped if there aren't two blank lines
        # at the end, so we look for that and fix it here.
        if ($notes =~ /(.+?)(?:\n(?!\n))$/) {
            print " " x 8, "<li>$1</li>\n";
        }
    }
    # Otherwise, assume items are delimited by \n
    else {
        while ($notes =~ m@(.*?)\n+@msg) {
            print " " x 8, "<li>$1</li>\n";
        }
    }

    print <<RELTAIL;
      </ul>
    </p:changelog>
  </p:release>
RELTAIL
}
print <<TAIL;
</p:project>
TAIL

exit;
