/*
 *  Copyright 2006-2022 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/**
 *  @internal
 *
 *  yafDPIPlugin.h
 *  Declares functions used by DPI plugins.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Chris Inacio, Emily Sarneso, Dillon Lareau
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 3.0.0
 *
 *  Copyright 2022 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Released under a GNU GPL 2.0-style license, please see license.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
 *  Trademark Office by Carnegie Mellon University.
 *
 *  This Software includes and/or makes use of Third-Party Software subject
 *  to its own license.
 *
 *  DM22-0007
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

/**
 *  @file yafDPIPlugin.h
 *
 *  header file for all of the functions in yafdpi.c needed by DPI plugins.
 */

#include <yaf/autoinc.h>

#if YAF_ENABLE_APPLABEL
#ifndef _YAF_DPI_PUBLIC_H_
#define _YAF_DPI_PUBLIC_H_

#include <ctype.h>
#include <glib/gstdio.h>
#include <pcre.h>

#if YAF_ENABLE_DPI
#define YAF_INT_PADDING_FLAG    0x01

/* max num of DPI fields we'll export - total */
#define YAF_MAX_CAPTURE_FIELDS  50
/* per side */
#define YAF_MAX_CAPTURE_SIDE    25
#endif  /* YAF_ENABLE_DPI */

typedef struct yfDPIContext_st {
    GHashTable  *dpiActiveHash;
    uint16_t     dpi_user_limit;
    uint16_t     dpi_total_limit;
    gboolean     dpiInitialized;
    gboolean     dpiApplabelOnly;
} yfDPIContext_t;

/**
 * A YAF Deep Packet Inspection Structure.  Holds offsets in the payload as to
 * important stuff that we want to capture (see protocol PCRE rule files)
 *
 */

typedef struct yfDPIData_st {
    /* offset in the payload to the good stuff */
    unsigned int   dpacketCapt;
    /* id of the field we found */
    uint16_t       dpacketID;
    /* length of good stuff */
    uint16_t       dpacketCaptLen;
} yfDPIData_t;

typedef struct ypDPIFlowCtx_st {
    /* this plugin's yaf context */
    yfDPIContext_t  *yfctx;
    yfDPIData_t     *dpi;
    /* keep track of how much we're exporting per flow */
    size_t           dpi_len;
    /* For Bi-Directional - need to know how many in fwd payload */
    uint8_t          captureFwd;
    /* Total Captures Fwd & Rev */
    uint8_t          dpinum;
    /* Primarily for Uniflow - Since we don't know if it's a FWD or REV flow
     * this is set to know where to start in the dpi array */
    uint8_t          startOffset;
    /* For Lists - we need to keep a ptr around so we can free it after
     * fBufAppend */
    void            *rec;
    /* extra buffer mainly for DNS stuff for now */
    uint8_t         *exbuf;
} ypDPIFlowCtx_t;


/*
 *  Defines the prototype signature of the function that each appLabel plug-in
 *  function must define.  The function scans the payload and returns an
 *  appLabel or returns 0 if the payload does not match its rules.
 *
 *  The function's parameters are:
 *
 *  -- payload the packet payload
 *  -- payloadSize size of the packet payload
 *  -- flow a pointer to the flow state structure
 *  -- val a pointer to biflow state (used for forward vs reverse)
 *
 */
uint16_t
ydpScanPayload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val);

/*
 *  The type of the ydpScanPayload() function.
 */
typedef uint16_t (*ydpScanPayload_fn)(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val);

/*
 *  Defines the prototype signature of an optional function that each appLabel
 *  plug-in may define.  The function is called when the plugin is first
 *  loaded and allows the plugin to initialize itself.
 *
 *  The function is passed the name and the arguments that were provided in
 *  the configuration file.  This is the only time the plug-in has access to
 *  the arguments, and the plug-in must use them at this time or store the
 *  them for later use.
 *
 *  The configuration file allows multiple applabel rules to reference the
 *  same plugin.  When that happens, this function is called for each.  The
 *  plugin can use the applabel parameter to know which applabel is being
 *  initialized.
 *
 *  The function should return a positive value if it is ready to process
 *  data, zero if the plug-in wants to be ignored, and a negative value to
 *  indicate an error.  When returning a negative value, the plug-in should
 *  also provide an error message via the `err` parameter; if not, a generic
 *  error message is generated.
 *
 *  The function's parameters are:
 *
 *  -- argc number of string arguments in argv (always >= 1)
 *  -- argv string arguments for this plugin (the first is the library name)
 *  -- applabel the number of the applabel (port) that is being initialized
 *  -- applabelOnly TRUE if only application labeling was requested
 *  -- err a parameter used to return error information back to the caller
 *
 */
int
ydpInitialize(
    int        argc,
    char      *argv[],
    uint16_t   applabel,
    gboolean   applabelOnly,
    GError   **err);

/*
 *  The type of the ydpInitialize() function.
 */
typedef int (*ydpInitialize_fn)(
    int        argc,
    char      *argv[],
    uint16_t   applabel,
    gboolean   applabelOnly,
    GError   **err);


/**
 *  Calls pcre_compile() on `regex` with `options` and returns the result.  If
 *  the compilation fails, a g_warning() is logged and function returns NULL.
 *
 *  @param regex The PCRE regular expression to be compiled
 *  @param options The PCRE flags to pass to pcre_compile()
 */
pcre *
ydPcreCompile(
    const char  *regex,
    int          options);


#if YAF_ENABLE_DPI

/*
 *  Defines the prototype signature of the function that each DPI plug-in must
 *  provide.
 */
void *
ydpProcessDPI(
    ypDPIFlowCtx_t       *flowContext,
    fbSubTemplateList_t  *stl,
    yfFlow_t             *flow,
    uint8_t               fwdcap,
    uint8_t               totalcap);

/*
 *  The type of the ydpProcessDPI() function.
 */
typedef void *(*ydpProcessDPI_fn)(
    ypDPIFlowCtx_t       *flowContext,
    fbSubTemplateList_t  *stl,
    yfFlow_t             *flow,
    uint8_t               fwdcap,
    uint8_t               totalcap);

/*
 *  Defines the prototype signature of the function that each DPI plug-in must
 *  provide.
 */
gboolean
ydpAddTemplates(
    fbSession_t  *session);

/*
 *  The type of the ydpAddTemplates() function.
 */
typedef gboolean (*ydpAddTemplates_fn)(
    fbSession_t  *session);

/*
 *  Defines the prototype signature of the function that each DPI plug-in must
 *  provide.
 */
void
ydpFreeRec(
    ypDPIFlowCtx_t  *flowContext);

/*
 *  The type of the ydpFreeRec() function.
 */
typedef void (*ydpFreeRec_fn)(
    ypDPIFlowCtx_t  *flowContext);

fbInfoModel_t *
ydGetDPIInfoModel(
    void);

void
ydRunPluginRegex(
    yfFlow_t       *flow,
    const uint8_t  *pkt,
    size_t          caplen,
    pcre           *expression,
    uint16_t        offset,
    uint16_t        elementID,
    uint16_t        applabel);

uint16_t
ydInitTemplate(
    fbTemplate_t        **newTemplate,
    fbSession_t          *session,
    fbInfoElementSpec_t  *spec,
    fbTemplateInfo_t     *mdInfo,
    uint16_t              tid,
    uint32_t              flags);

void *
ydProcessGenericPlugin(
    ypDPIFlowCtx_t       *flowContext,
    fbSubTemplateList_t  *stl,
    yfFlow_t             *flow,
    uint8_t               fwdcap,
    uint8_t               totalcap,
    uint16_t              stlTID,
    const fbTemplate_t   *stlTemplate,
    const char           *blIEName);

const fbInfoElement_t *
ydLookupNamedBlByName(
    const fbInfoElement_t  *ie);

const fbInfoElement_t *
ydLookupNamedBlByID(
    uint32_t   ent,
    uint16_t   id);

#endif  /* YAF_ENABLE_DPI */
#endif /* _YAF_DPI_PUBLIC_H_ */
#endif /* #if YAF_ENABLE_APPLABEL */
