/*
 *  Copyright 2007-2022 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/**
 *  @internal
 *
 *  @file aolplugin.c
 *
 *
 *  This tries to recognize the AOL instant Messenger (OSCAR) protocol
 *  http://en.wikipedia.org/wiki/OSCAR_protocol
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 3.0.0
 *
 *  Copyright 2022 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Released under a GNU GPL 2.0-style license, please see license.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
 *  Trademark Office by Carnegie Mellon University.
 *
 *  This Software includes and/or makes use of Third-Party Software subject
 *  to its own license.
 *
 *  DM22-0007
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <yaf/yafDPIPlugin.h>

#define AIM_PORT_NUMBER 5190

/* Local Prototypes */

static uint16_t
getTLVID(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    uint16_t        offsetptr);


/**
 * ydpScanPayload
 *
 * the scanner for recognizing aol instant messenger/ICQ  packets
 *
 * @param payload the packet payload
 * @param payloadSize size of the packet payload
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 *
 * @return aim_port_number
 *         otherwise 0
 */
uint16_t
ydpScanPayload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    gboolean snac = FALSE;
    uint16_t flap_seq_number = 0;
    uint16_t offsetptr = 0;
    uint16_t flap_data_size = 0;
    uint8_t  class;
    uint16_t tlv_id;

    if (payloadSize < 6) {
        return 0;
    }

    if (*(payload + offsetptr) != 0x2a) {
        return 0;
    }
    offsetptr++;

    class = *(payload + offsetptr);
    if ((class == 0) || (class > 5)) {
        return 0;
    }

    if (class == 2) {
        /* SNAC data */
        snac = TRUE;
    }

    offsetptr++;
    /* seq number */

    flap_seq_number = ntohs(*(uint16_t *)(payload + offsetptr));
    if (flap_seq_number > 0xEFFF) {
        return 0;
    }

    offsetptr += 2;
    /* size of data */
    flap_data_size = ntohs(*(uint16_t *)(payload + offsetptr));
    offsetptr += 2;

    if (snac) {
        uint16_t family;
        uint16_t family_sub_id;

        if ((size_t)offsetptr + 4 > payloadSize) {
            return 0;
        }

        family = ntohs(*(uint16_t *)(payload + offsetptr));
        if (family > 0x17 && family != 0x85) {
            return 0;
        }

        offsetptr += 2;

        family_sub_id = ntohs(*(uint16_t *)(payload + offsetptr));
        /* there are more detailed specifications on what family id and
         * family_sub_id can be paired, but too many to efficiently check
         * so we will generalize */
        if (family_sub_id > 0x21) {
            return 0;
        }

        offsetptr += 8; /* 2 for SNAC flags, 4 for request ID */

        if (offsetptr > payloadSize) {
            return 0;
        }
    }

    if (class == 1) {
        uint32_t protocol;

        /* protocol version */
        if ((size_t)offsetptr + 4 > payloadSize) {
            return 0;
        }

        protocol = ntohl(*(uint32_t *)(payload + offsetptr));

        if (protocol > 1) {
            return 0;
        }

        offsetptr += 4;
        if (flap_data_size != 4) {
            tlv_id = getTLVID(payload, payloadSize, offsetptr);
            if (tlv_id != 6 && tlv_id != 7 && tlv_id != 8 && tlv_id != 3 &&
                tlv_id != 148 && tlv_id != 74)
            {
                return 0;
            }
            offsetptr += 2;
        }
    }

    return AIM_PORT_NUMBER;
}


static uint16_t
getTLVID(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    uint16_t        offsetptr)
{
    uint16_t tlvid;

    if ((size_t)offsetptr + 2 > payloadSize) {
        return 0;
    }

    tlvid = ntohs(*(uint16_t *)(payload + offsetptr));

    return tlvid;
}
