/*
 *  Copyright 2007-2022 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/**
 *  @internal
 *
 *  @file dhcpplugin.c
 *
 *
 *  This tries to recognize the DHCP protocol
 *  rfc 2131
 *
 *  The Dynamic Host Configuration Protocol (DHCP) provides a framework
 *  for passing configuration information to hosts on a TCPIP network.
 *  It is based on the Bootstrap Protocol (BOOTP) adding the add'l
 *  capability of automatic allocation of reusable network addresses
 *  and add'l config options.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 3.0.0
 *
 *  Copyright 2022 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Released under a GNU GPL 2.0-style license, please see license.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
 *  Trademark Office by Carnegie Mellon University.
 *
 *  This Software includes and/or makes use of Third-Party Software subject
 *  to its own license.
 *
 *  DM22-0007
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <yaf/yafDPIPlugin.h>

#define DHCP_PORT_NUMBER 67
#define MAGICCOOKIE 0x63825363

/**
 * ydpScanPayload
 *
 * the scanner for recognizing DHCP packets
 *
 * @param payload the packet payload
 * @param payloadSize size of the packet payload
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 *
 * @return dhcp port number
 *         otherwise 0
 */
uint16_t
ydpScanPayload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    uint8_t  op, htype;
    uint16_t flags, offsetptr = 0;
    uint32_t magic_cookie;
    int      loop;

    if (payloadSize < 44) {
        return 0;
    }
    if (flow->key.proto != YF_PROTO_UDP) {
        return 0;
    }

    /* MESSAGE TYPE */
    op = payload[0];
    if (op != 2 && op != 1) {
        return 0;   /* BOOTREPLY = 2, BOOTREQUEST = 1 */
    }
    offsetptr++;

    /* Hardware type */
    htype = *(payload + offsetptr);
    if (htype != 1) {
        return 0;
    }

    /* hardware len is after type */

    offsetptr += 2;

    /* hops should be 0 */
    if (*(payload + offsetptr) != 0) {
        return 0;
    }

    /* transaction ID next & then seconds elapsed */
    offsetptr += 7;

    flags = ntohs(*(uint16_t *)(payload + offsetptr));
    if (flags != 0x8000 && flags != 0) {
        return 0;  /* only 1 (Broadcast flag) bit can be set) */
    }

    /* client addr is after flags - can be different based on type of message
     * */
    offsetptr += 6;

    if (op == 1) {
        /* yiaddr, siaddr, and giaddr should be 0 */
        for (loop = 0; loop < 12; loop++) {
            if (*(payload + offsetptr + loop) != 0) {
                return 0;
            }
        }
    }
    /* 12 for above yiaddr, siaddr, and giaddr, 16 for chaddr */
    offsetptr += 28;
    /* 64 for sname, 128 for file, 4 for magic cookie */
    if ((size_t)offsetptr + 196 <= payloadSize) {
        offsetptr += 192;
    } else {
        /* should be good enough - but magic cookie will secure the decision */
        return DHCP_PORT_NUMBER;
    }

    magic_cookie = ntohl(*(uint32_t *)(payload + offsetptr));
    if (magic_cookie != MAGICCOOKIE) {
        return 0;
    }

    offsetptr += 4;
    if (offsetptr >= payloadSize) {
        /* just enough */
        return DHCP_PORT_NUMBER;
    }

    /* OPTIONS SECTION! */

    return DHCP_PORT_NUMBER;
}
