/*
 *  Copyright 2007-2022 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/**
 *  @internal
 *
 *  @file dnsplugin.c
 *
 *  provides a plugin to the ipfix payload classifier to attempt to determine
 *  if a packet payload is a DNS packet (see RFC 1035)
 *
 *  @note defining PAYLOAD_INSPECTION at compile time will attempt to better
 *  inspection of the packet payload at a cost of deeper inspection;  even with
 *  PAYLOAD_INSPECTION enabled, it is possible that this may not be 100%
 *  correct in ID'ing the packets
 *
 *  ------------------------------------------------------------------------
 *  Authors: Chris Inacio
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 3.0.0
 *
 *  Copyright 2022 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Released under a GNU GPL 2.0-style license, please see license.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
 *  Trademark Office by Carnegie Mellon University.
 *
 *  This Software includes and/or makes use of Third-Party Software subject
 *  to its own license.
 *
 *  DM22-0007
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */
#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>

#include <yaf/yafDPIPlugin.h>
#if YAF_ENABLE_DPI

/**
 *  Pulls (reads) an integer from a data buffer at an offset and advances the
 *  offset pointer.  Sets the offset to the data length when there is not
 *  enough data (or offset already excceds the data length).  Does not modify
 *  the destination when there is not enough data available.
 *
 *  @param r_type the type of integer to read (uint8_t, uint16_t, etc)
 *  @param r_swap macro to swap network/host endian
 *  @param r_dst  where to put the value, a pointer
 *  @param r_off  the offset, a pointer
 *  @param r_data the data buffer to pull the value from
 *  @param r_len  the length of the data buffer
 */
#define READ_TYPE_INC(r_type, r_swap, r_dst, r_off, r_data, r_len)      \
    if ((size_t)(*(r_off)) + sizeof(r_type) <= (r_len)) {               \
        *(r_dst) = r_swap(*((r_type *)((r_data) + *(r_off))));          \
        *(r_off) += sizeof(r_type);                                     \
    } else {                                                            \
        *(r_off) = (r_len);                                             \
    }

/**
 *  Pull (read) a uint8_t from the data buffer `r_data` of length `r_len` at
 *  the offset `r_off` (a pointer), store the value at `r_dst` (a pointer),
 *  and advance the offset pointer.
 */
#define READ_U8_INC(r_dst, r_off, r_data, r_len)                \
    READ_TYPE_INC(uint8_t, , r_dst, r_off, r_data, r_len)

/**
 *  Pull (read) a uint16_t from the data buffer `r_data` of length `r_len` at
 *  the offset `r_off` (a pointer), store the value at `r_dst` (a pointer),
 *  and advance the offset pointer.
 */
#define READ_U16_INC(r_dst, r_off, r_data, r_len)               \
    READ_TYPE_INC(uint16_t, ntohs, r_dst, r_off, r_data, r_len)

/**
 *  Pull (read) a uint32_t from the data buffer `r_data` of length `r_len` at
 *  the offset `r_off` (a pointer), store the value at `r_dst` (a pointer),
 *  and advance the offset pointer.
 */
#define READ_U32_INC(r_dst, r_off, r_data, r_len)               \
    READ_TYPE_INC(uint32_t, ntohl, r_dst, r_off, r_data, r_len)



/* DNS QR. Level 1 #1 */
#define YAF_DNS_FLOW_TID                0xCF00
#define YAF_DNS_FLOW_NAME               "yaf_dns_rr"
#define YAF_DNS_FLOW_DESC               NULL

/* DNS A. Level 2 #1 */
#define YAF_DNS_A_FLOW_TID              0xCE01
#define YAF_DNS_A_FLOW_NAME             "yaf_dns_a"
#define YAF_DNS_A_FLOW_DESC             NULL

/* DNS AAAA. Level 2 #2 */
#define YAF_DNS_AAAA_FLOW_TID           0xCE02
#define YAF_DNS_AAAA_FLOW_NAME          "yaf_dns_aaaa"
#define YAF_DNS_AAAA_FLOW_DESC          NULL

/* DNS CNAME. Level 2 #3 */
#define YAF_DNS_CNAME_FLOW_TID          0xCE03
#define YAF_DNS_CNAME_FLOW_NAME         "yaf_dns_cname"
#define YAF_DNS_CNAME_FLOW_DESC         NULL

/* DNS MX. Level 2 #4 */
#define YAF_DNS_MX_FLOW_TID             0xCE04
#define YAF_DNS_MX_FLOW_NAME            "yaf_dns_mx"
#define YAF_DNS_MX_FLOW_DESC            NULL

/* DNS NS. Level 2 #5 */
#define YAF_DNS_NS_FLOW_TID             0xCE05
#define YAF_DNS_NS_FLOW_NAME            "yaf_dns_ns"
#define YAF_DNS_NS_FLOW_DESC            NULL

/* DNS PTR. Level 2 #6 */
#define YAF_DNS_PTR_FLOW_TID            0xCE06
#define YAF_DNS_PTR_FLOW_NAME           "yaf_dns_ptr"
#define YAF_DNS_PTR_FLOW_DESC           NULL

/* DNS TXT. Level 2 #7 */
#define YAF_DNS_TXT_FLOW_TID            0xCE07
#define YAF_DNS_TXT_FLOW_NAME           "yaf_dns_txt"
#define YAF_DNS_TXT_FLOW_DESC           NULL

/* DNS SRV. Level 2 #8 */
#define YAF_DNS_SRV_FLOW_TID            0xCE08
#define YAF_DNS_SRV_FLOW_NAME           "yaf_dns_srv"
#define YAF_DNS_SRV_FLOW_DESC           NULL

/* DNS SOA. Level 2 #9 */
#define YAF_DNS_SOA_FLOW_TID            0xCE09
#define YAF_DNS_SOA_FLOW_NAME           "yaf_dns_soa"
#define YAF_DNS_SOA_FLOW_DESC           NULL

/* DNS DS. Level 2 #10 */
#define YAF_DNS_DS_FLOW_TID             0xCE0E
#define YAF_DNS_DS_FLOW_NAME            "yaf_dns_ds"
#define YAF_DNS_DS_FLOW_DESC            NULL

/* DNS RRSIG. Level 2 #11*/
#define YAF_DNS_RRSIG_FLOW_TID          0xCE0F
#define YAF_DNS_RRSIG_FLOW_NAME         "yaf_dns_rrsig"
#define YAF_DNS_RRSIG_FLOW_DESC         NULL

/* DNS NSEC. Level 2 #12 */
#define YAF_DNS_NSEC_FLOW_TID           0xCE11
#define YAF_DNS_NSEC_FLOW_NAME          "yaf_dns_nsec"
#define YAF_DNS_NSEC_FLOW_DESC          NULL

/* DNS KEY. Level 2 #13 */
#define YAF_DNS_DNSKEY_FLOW_TID         0xCE12
#define YAF_DNS_DNSKEY_FLOW_NAME        "yaf_dns_dnskey"
#define YAF_DNS_DNSKEY_FLOW_DESC        NULL

/* DNS NSEC3. Level 2 #14 */
#define YAF_DNS_NSEC3_FLOW_TID          0xCE13
#define YAF_DNS_NSEC3_FLOW_NAME         "yaf_dns_nsec3"
#define YAF_DNS_NSEC3_FLOW_DESC         NULL

/* DNS NSEC3PARM. Level 2 #15 */
#define YAF_DNS_NSEC3PARAM_FLOW_TID     0xCE15
#define YAF_DNS_NSEC3PARAM_FLOW_NAME    "yaf_dns_nsec3param"
#define YAF_DNS_NSEC3PARAM_FLOW_DESC    NULL


/* SM has an Exact Match version. Changes won't break, but sync is good */
static fbInfoElementSpec_t dns_spec[] = {
    {"dnsDetailRecordList",     FB_IE_VARLEN, 0 }, /*based on type of RR */
    /* used by SM to label DNS Resource Record */
    {"dnsName",                 FB_IE_VARLEN, 0 },
    {"dnsTTL",                  4, 0 },
    /* used by SM to label DNS Resource Record */
    {"dnsRRType",               2, 0 },
    {"dnsQueryResponse",        1, 0 },  /*Q(0) or R(1) - uint8*/
    {"dnsAuthoritative",        1, 0 },  /* authoritative response (1)*/
    {"dnsResponseCode",         1, 0 },  /* nxdomain (1) */
    {"dnsSection",              1, 0 },  /*0, 1, 2, 3 (q, ans, auth, add'l) */
    {"dnsId",                   2, 0 },
    {"paddingOctets",           4, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

typedef struct dns_flow_st {
    fbSubTemplateList_t   dnsRRList;
    fbVarfield_t          dnsName;
    uint32_t              dnsTTL;
    uint16_t              dnsRRType;
    uint8_t               dnsQueryResponse;
    uint8_t               dnsAuthoritative;
    uint8_t               dnsResponseCode;
    uint8_t               dnsSection;
    uint16_t              dnsId;
    uint8_t               padding[4];
} dns_flow_t;


static fbInfoElementSpec_t dns_a_spec[] = {
    {"dnsA",         4, 0 },
    FB_IESPEC_NULL
};

typedef struct dns_a_flow_st {
    uint32_t   dnsA;
} dns_a_flow_t;

static fbInfoElementSpec_t dns_aaaa_spec[] = {
    {"dnsAAAA",         16, 0 },
    FB_IESPEC_NULL
};

typedef struct dns_aaaa_flow_st {
    uint8_t   dnsAAAA[16];
} dns_aaaa_flow_t;

static fbInfoElementSpec_t dns_cname_spec[] = {
    {"dnsCNAME",                  FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

typedef struct dns_cname_flow_st {
    fbVarfield_t   dnsCNAME;
} dns_cname_flow_t;

static fbInfoElementSpec_t dns_mx_spec[] = {
    {"dnsMXExchange",             FB_IE_VARLEN, 0 },
    {"dnsMXPreference",           2, 0 },
    {"paddingOctets",             6, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

typedef struct dns_mx_flow_st {
    fbVarfield_t   dnsMXExchange;
    uint16_t       dnsMXPreference;
    uint8_t        padding[6];
} dns_mx_flow_t;

static fbInfoElementSpec_t dns_ns_spec[] = {
    {"dnsNSDName",                FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

typedef struct dns_ns_flow_st {
    fbVarfield_t   dnsNSDName;
} dns_ns_flow_t;

static fbInfoElementSpec_t dns_ptr_spec[] = {
    {"dnsPTRDName",               FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

typedef struct dns_ptr_flow_st {
    fbVarfield_t   dnsPTRDName;
} dns_ptr_flow_t;

static fbInfoElementSpec_t dns_txt_spec[] = {
    {"dnsTXTData",                FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

typedef struct dns_txt_flow_st {
    fbVarfield_t   dnsTXTData;
} dns_txt_flow_t;

static fbInfoElementSpec_t dns_soa_spec[] = {
    {"dnsSOAMName",               FB_IE_VARLEN, 0 },
    {"dnsSOARName",               FB_IE_VARLEN, 0 },
    {"dnsSOASerial",              4, 0 },
    {"dnsSOARefresh",             4, 0 },
    {"dnsSOARetry",               4, 0 },
    {"dnsSOAExpire",              4, 0 },
    {"dnsSOAMinimum",             4, 0 },
    {"paddingOctets",             4, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

typedef struct dns_soa_flow_st {
    fbVarfield_t  dnsSOAMName;
    fbVarfield_t  dnsSOARName;
    uint32_t      dnsSOASerial;
    uint32_t      dnsSOARefresh;
    uint32_t      dnsSOARetry;
    uint32_t      dnsSOAExpire;
    uint32_t      dnsSOAMinimum;
    uint8_t       padding[4];
} dns_soa_flow_t;

static fbInfoElementSpec_t dns_srv_spec[] = {
    {"dnsSRVTarget",              FB_IE_VARLEN, 0 },
    {"dnsSRVPriority",            2, 0 },
    {"dnsSRVWeight",              2, 0 },
    {"dnsSRVPort",                2, 0 },
    {"paddingOctets",             2, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

typedef struct dns_srv_flow_st {
    fbVarfield_t   dnsSRVTarget;
    uint16_t       dnsSRVPriority;
    uint16_t       dnsSRVWeight;
    uint16_t       dnsSRVPort;
    uint8_t        padding[2];
} dns_srv_flow_t;


static fbInfoElementSpec_t dns_ds_spec[] = {
    {"dnsDSDigest",               FB_IE_VARLEN, 0 },
    {"dnsDSKeyTag",               2, 0 },
    {"dnsDSAlgorithm",            1, 0 },
    {"dnsDSDigestType",           1, 0 },
    {"paddingOctets",             4, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

typedef struct dns_ds_flow_st {
    fbVarfield_t   dnsDSDigest;
    uint16_t       dnsDSKeyTag;
    uint8_t        dnsDSAlgorithm;
    uint8_t        dnsDSDigestType;
    uint8_t        padding[4];
} dns_ds_flow_t;


static fbInfoElementSpec_t dns_rrsig_spec[] = {
    {"dnsRRSIGSigner",              FB_IE_VARLEN, 0 },
    {"dnsRRSIGSignature",           FB_IE_VARLEN, 0 },
    {"dnsRRSIGSignatureInception",  4, 0 },
    {"dnsRRSIGSignatureExpiration", 4, 0 },
    {"dnsRRSIGOriginalTTL",         4, 0 },
    {"dnsRRSIGKeyTag",              2, 0 },
    {"dnsRRSIGTypeCovered",         2, 0 },
    {"dnsRRSIGAlgorithm",           1, 0 },
    {"dnsRRSIGLabels",              1, 0 },
    {"paddingOctets",               6, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

typedef struct dns_rrsig_flow_st {
    fbVarfield_t   dnsRRSIGSigner;
    fbVarfield_t   dnsRRSIGSignature;
    uint32_t       dnsRRSIGSignatureInception;
    uint32_t       dnsRRSIGSignatureExpiration;
    uint32_t       dnsRRSIGOriginalTTL;
    uint16_t       dnsRRSIGKeyTag;
    uint16_t       dnsRRSIGTypeCovered;
    uint8_t        dnsRRSIGAlgorithm;
    uint8_t        dnsRRSIGLabels;
    uint8_t        padding[6];
} dns_rrsig_flow_t;

static fbInfoElementSpec_t dns_nsec_spec[] = {
    {"dnsNSECNextDomainName",     FB_IE_VARLEN, 0 },
    {"dnsNSECTypeBitMaps",        FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

typedef struct dns_nsec_flow_st {
    fbVarfield_t    dnsNSECNextDomainName;
    fbVarfield_t    dnsNSECTypeBitMaps;
} dns_nsec_flow_t;

static fbInfoElementSpec_t dns_dnskey_spec[] = {
    {"dnsDNSKEYPublicKey",        FB_IE_VARLEN, 0 },
    {"dnsDNSKEYFlags",            2, 0 },
    {"dnsDNSKEYProtocol",         1, 0 },
    {"dnsDNSKEYAlgorithm",        1, 0 },
    {"paddingOctets",             4, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

typedef struct dns_dnskey_flow_st {
    fbVarfield_t   dnsDNSKEYPublicKey;
    uint16_t       dnsDNSKEYFlags;
    uint8_t        dnsDNSKEYProtocol;
    uint8_t        dnsDNSKEYAlgorithm;
    uint8_t        padding[4];
} dns_dnskey_flow_t;

static fbInfoElementSpec_t dns_nsec3_spec[] = {
    {"dnsNSEC3Salt",                FB_IE_VARLEN, 0 },
    {"dnsNSEC3NextHashedOwnerName", FB_IE_VARLEN, 0 },
    {"dnsNSEC3TypeBitMaps",         FB_IE_VARLEN, 0 },
    {"dnsNSEC3Iterations",          2, 0 },
    {"dnsNSEC3Algorithm",           1, 0 },
    {"dnsNSEC3Flags",               1, 0 },
    {"paddingOctets",               4, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

typedef struct dns_nsec3_flow_st {
    fbVarfield_t   dnsNSEC3Salt;
    fbVarfield_t   dnsNSEC3NextHashedOwnerName;
    fbVarfield_t   dnsNSEC3TypeBitMaps;
    uint16_t       dnsNSEC3Iterations;
    uint8_t        dnsNSEC3Algorithm;
    uint8_t        dnsNSEC3Flags;
    uint8_t        padding[4];
} dns_nsec3_flow_t;


static fbInfoElementSpec_t dns_nsec3param_spec[] = {
    {"dnsNSEC3PARAMSalt",         FB_IE_VARLEN, 0 },
    {"dnsNSEC3PARAMIterations",   2, 0 },
    {"dnsNSEC3PARAMAlgorithm",    1, 0 },
    {"dnsNSEC3PARAMFlags",        1, 0 },
    {"paddingOctets",             4, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

typedef struct dns_nsec3param_flow_st {
    fbVarfield_t   dnsNSEC3PARAMSalt;
    uint16_t       dnsNSEC3PARAMIterations;
    uint8_t        dnsNSEC3PARAMAlgorithm;
    uint8_t        dnsNSEC3PARAMFlags;
    uint8_t        padding[4];
} dns_nsec3param_flow_t;

static fbTemplate_t     *dns_template;
static fbTemplate_t     *dns_a_template;
static fbTemplate_t     *dns_aaaa_template;
static fbTemplate_t     *dns_cname_template;
static fbTemplate_t     *dns_mx_template;
static fbTemplate_t     *dns_ns_template;
static fbTemplate_t     *dns_ptr_template;
static fbTemplate_t     *dns_txt_template;
static fbTemplate_t     *dns_srv_template;
static fbTemplate_t     *dns_soa_template;
static fbTemplate_t     *dns_ds_template;
static fbTemplate_t     *dns_nsec_template;
static fbTemplate_t     *dns_nsec3_template;
static fbTemplate_t     *dns_nsec3param_template;
static fbTemplate_t     *dns_rrsig_template;
static fbTemplate_t     *dns_dnskey_template;

gboolean dnssec_global = FALSE;

/* DNS Max Name length */
#define DNS_MAX_NAME_LENGTH     255
#endif  /* YAF_ENABLE_DPI */

typedef struct ycDnsScanMessageHeader_st {
    uint16_t   id;

    uint16_t   qr     : 1;
    uint16_t   opcode : 4;
    uint16_t   aa     : 1;
    uint16_t   tc     : 1;
    uint16_t   rd     : 1;
    uint16_t   ra     : 1;
    uint16_t   z      : 1;
    uint16_t   ad     : 1;
    uint16_t   cd     : 1;
    uint16_t   rcode  : 4;

    uint16_t   qdcount;
    uint16_t   ancount;
    uint16_t   nscount;
    uint16_t   arcount;
} ycDnsScanMessageHeader_t;

#define DNS_PORT_NUMBER 53
#define DNS_NAME_COMPRESSION 0xc0
#define DNS_NAME_OFFSET 0x0FFF
#define DNS_MAX_NAME_LENGTH 255
/** this field defines the number of octects we fuzz the size of the
 *  DNS to the IP+TCP+payload size with; we don't record any TCP
 *  options, so it is possible to have a few extra bytes in the
 *  calculation, and we won't say that's bad until that is larger
 *  than the following constant */
#define DNS_TCP_FLAG_SLACK 8

/** Since NETBIOS looks A LOT like DNS, there's no need to create
 *  a separate plugin for it - if we think it's NETBIOS we will
 *  return NETBIOS_PORT */
#define NETBIOS_PORT 137

#define PAYLOAD_INSPECTION 1

/**
 * local prototypes
 *
 */

#ifdef PAYLOAD_INSPECTION
static uint16_t
ycDnsScanCheckResourceRecord(
    const uint8_t  *payload,
    unsigned int   *offset,
    unsigned int    payloadSize);

#endif /* ifdef PAYLOAD_INSPECTION */

#if YAF_ENABLE_DPI
static void
ypDNSParser(
    dns_flow_t    **dnsRecord,
    yfFlow_t       *flow,
    yfFlowVal_t    *val,
    uint8_t        *buf,
    unsigned int   *bufLen,
    uint8_t         recordCount,
    uint16_t        export_limit);

static uint16_t
ypDnsScanResourceRecord(
    dns_flow_t    **dnsRecord,
    const uint8_t  *payload,
    unsigned int    payloadSize,
    uint16_t       *offset,
    uint8_t        *buf,
    unsigned int   *bufLen,
    uint16_t        export_limit);

static uint8_t
ypGetDNSQName(
    uint8_t        *buf,
    uint16_t        bufoffset,
    const uint8_t  *payload,
    unsigned int    payloadSize,
    uint16_t       *offset,
    uint16_t        export_limit);
#endif  /* YAF_ENABLE_DPI */

static void
ycDnsScanRebuildHeader(
    const uint8_t             *payload,
    ycDnsScanMessageHeader_t  *header);


/**
 * ydpScanPayload
 *
 * scans a payload to determine if the payload is a dns request/reply.
 * It checks the structure for self referential integrity, but it can't
 * guarantee that the payload is actually DNS, it could be
 * some degenerate random data
 *
 * name abomination has been achieved by combining multiple naming standards
 * until the prefix to
 * the function name is dnsplugin_LTX_ycDnsScan --- it's a feature
 *
 * @param payload pointer to the payload data
 * @param payloadSize the size of the payload parameter
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 * @return 0 for no match DNS_PORT_NUMBER (53) for a match
 *
 */
uint16_t
ydpScanPayload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    unsigned int loop = 0;
    uint16_t     msglen;
    uint16_t     firstpkt = payloadSize;
    ycDnsScanMessageHeader_t header;
    gboolean     netbios = FALSE;
    unsigned int payloadOffset;
    uint16_t     qtype = 0;
#if YAF_ENABLE_DPI
    unsigned int recordCount = 0;
    uint16_t     direction;
#endif

    if (payloadSize < sizeof(ycDnsScanMessageHeader_t)) {
        /*fprintf(stderr, " <dns exit 1> ");
         * g_debug("returning at line 118");*/
        return 0;
    }

    if (flow->key.proto == YF_PROTO_TCP) {
        while (loop < val->pkt && loop < YAF_MAX_PKT_BOUNDARY) {
            if (val->paybounds[loop] == 0) {
                loop++;
            } else {
                firstpkt = val->paybounds[loop];
                break;
            }
        }
        msglen = ntohs(*((uint16_t *)(payload)));
        if ((msglen + 2) == firstpkt) {
            /* this is the weird message length in TCP */
            payload += sizeof(uint16_t);
            payloadSize -= sizeof(uint16_t);
        }
    }

    ycDnsScanRebuildHeader(payload, &header);

    if ((header.opcode > 5) || (header.opcode == 3)) {
        if (header.opcode >= 5 && header.opcode < 9) {
            netbios = TRUE;
        } else {
            return 0;
        }
    }

    /* rfc 2136 updates rfc 1035 */
    /* 16-22 are DNSSEC rcodes*/
    if ((header.rcode > 10) && (1 == header.qr)) {
        if ((header.rcode < 16) || (header.rcode > 22)) {
            /*g_debug("returning at line 197 %d", header.rcode);*/
            return 0;
        }
    }

    /* rfc states that Z is reserved for future use and must be zero */
    if (0 != header.z) {
        /*g_debug("returning at line 141");*/
        return 0;
    }

    /* check to make sure resource records are not empty -
     * gets rid of all 0's payloads */
    if (header.qdcount == 0 && header.ancount == 0 && header.nscount == 0
        && header.arcount == 0)
    {
        if (!(header.rcode > 0 && header.qr == 1)) {
            /* DNS responses that are not errors will have something in them*/
            return 0;
        }
    }

    /* query validation */
    if (header.qr == 0) {
        if ((header.rcode > 0 || header.aa != 0 || header.ra != 0 ||
             header.ad != 0))
        {
            /* queries should not have an rcode, an authoritative answer,
             * recursion available, or authenticated data */
            return 0;
        }
        if (!(header.qdcount > 0)) {
            /* queries should have at least one question */
            return 0;
        }
    }

#ifdef PAYLOAD_INSPECTION
    /* parse through the rest of the DNS message, only the header is fixed
     * in size */
    payloadOffset = sizeof(ycDnsScanMessageHeader_t);
    /* the the query entries */

    if (payloadOffset >= payloadSize) {
        return 0;
    }
    /*fprintf(stderr,"dns qdcount %d, ancount %d, nscount %d, arcount
     * %d\n",header.qdcount,header.ancount,header.nscount, header.arcount);*/

    for (loop = 0; loop < header.qdcount; loop++) {
        uint8_t  sizeOct = *(payload + payloadOffset);
        uint16_t qclass;
        uint8_t  comp = 0;            /* turn on if something is compressed */

        while (0 != sizeOct && payloadOffset < payloadSize) {
            if (DNS_NAME_COMPRESSION == (sizeOct & DNS_NAME_COMPRESSION)) {
                payloadOffset += sizeof(uint16_t);
                /* compression happened so we don't need add 1 later */
                comp = 1;
            } else {
                payloadOffset += sizeOct + 1;
            }
            if (payloadOffset >= payloadSize) {
                return 0;
            }
            sizeOct = *(payload + payloadOffset);
        }

        if (payloadOffset >= payloadSize) {
            /* this is either a DNS fragment, or a malformed DNS */
            /*fprintf(stderr, " <dns exit 5> ");*/
            return 0;
        }

        /* get past the terminating 0 length in the name if NO COMPRESSION*/
        if (!comp) {
            payloadOffset++;
        }

        if ((payloadOffset + 2) > payloadSize) {
            return 0;
        }

        /* check the query type */
#if HAVE_ALIGNED_ACCESS_REQUIRED
        qtype = ((*(payload + payloadOffset)) << 8) |
            ((*(payload + payloadOffset + 1)) );

        qtype = ntohs(qtype);
#else /* if HAVE_ALIGNED_ACCESS_REQUIRED */
        qtype = ntohs(*((uint16_t *)(payload + payloadOffset)));
#endif /* if HAVE_ALIGNED_ACCESS_REQUIRED */
        if (qtype == 0) {
            return 0;
        } else if (qtype > 52) {
            if ((qtype < 249) || (qtype > 253)) {
                if ((qtype != 32769) && (qtype != 32768) && (qtype != 99)) {
                    return 0;
                }
            }
        }

        if (qtype == 32) {
            netbios = TRUE;
        } else if (qtype == 33 && (flow->key.sp == NETBIOS_PORT ||
                                   flow->key.dp == NETBIOS_PORT))
        {
            netbios = TRUE;
        }

        payloadOffset += sizeof(uint16_t);

        if ((payloadOffset + 2) > payloadSize) {
            return 0;
        }

        /* check the class code */
#if HAVE_ALIGNED_ACCESS_REQUIRED
        qclass = ((*(payload + payloadOffset)) << 8) |
            ((*(payload + payloadOffset + 1)) );
        qclass = ntohs(qclass);
#else
        qclass = ntohs(*((uint16_t *)(payload + payloadOffset)));
#endif /* if HAVE_ALIGNED_ACCESS_REQUIRED */

        if (qclass > 4 && qclass != 255) {
            /*fprintf(stderr, " <dns exit 7, qclass = %d> ", qclass);*/
            return 0;
        }

        if (netbios) {
            if (qclass != 1) {
                return 0;
            }
        }

        payloadOffset += sizeof(uint16_t);

        if (payloadOffset > payloadSize) {
            return 0;
        }
    }

    /* check each record for the answer record count */
    for (loop = 0; loop < header.ancount; loop++) {
        uint16_t rc;

        rc = ycDnsScanCheckResourceRecord(payload, &payloadOffset,
                                          payloadSize);
        if (0 == rc) {
            return rc;
        }

        if (netbios && (rc != 1 && rc != 2 && rc != 10 && rc != 32 &&
                        rc != 33))
        {
            return 0;
        } else if (rc == 32) {
            netbios = TRUE;
        } else if (rc == 33 && header.qdcount == 0) {
            netbios = TRUE;
        }

#if YAF_ENABLE_DPI
        if (rc != 41) {
            recordCount++;
        }
#endif
    }

    /* check each record for the name server resource record count */
    for (loop = 0; loop < header.nscount; loop++) {
        uint16_t rc;
        rc = ycDnsScanCheckResourceRecord(payload, &payloadOffset,
                                          payloadSize);
        if (0 == rc) {
            return 0;
        }

        if (netbios && (rc != 1 && rc != 2 && rc != 10 && rc != 32 &&
                        rc != 33))
        {
            return 0;
        } else if (rc == 2 && header.qdcount == 0) {
            netbios = TRUE;
        }

#if YAF_ENABLE_DPI
        if (rc != 41) {
            recordCount++;
        }
#endif
    }
    /* check each record for the additional record count */
    for (loop = 0; loop < header.arcount; loop++) {
        uint16_t rc;
        rc = ycDnsScanCheckResourceRecord(payload, &payloadOffset,
                                          payloadSize);
        if (0 == rc) {
            return 0;
        }

        if (netbios && (rc != 1 && rc != 2 && rc != 10 && rc != 32 &&
                        rc != 33))
        {
            return 0;
        }

#if YAF_ENABLE_DPI
        if (rc != 41) {
            recordCount++;
        }
#endif
    }

    if (netbios) {
        return NETBIOS_PORT;
    }

#if YAF_ENABLE_DPI
    if (val == &(flow->val)) {
        direction = 0;
    } else {
        direction = 1;
    }

#if defined(YAF_ENABLE_DNSAUTH) && defined(YAF_ENABLE_DNSNXDOMAIN)
    if ((header.aa == 1) || (header.rcode == 3)) {
        if (recordCount + header.qdcount) {
            ydRunPluginRegex(flow, payload, 0, NULL,
                              (recordCount + header.qdcount), direction,
                              DNS_PORT_NUMBER);
        }
    }
#elif defined(YAF_ENABLE_DNSAUTH) && !defined(YAF_ENABLE_DNSNXDOMAIN)
    if (header.aa == 1) {
        if (recordCount + header.qdcount) {
            ydRunPluginRegex(flow, payload, 0, NULL,
                              (recordCount + header.qdcount),
                              direction, DNS_PORT_NUMBER);
        }
    }
#elif defined(YAF_ENABLE_DNSNXDOMAIN) && !defined(YAF_ENABLE_DNSAUTH)
    if (header.rcode == 3) {
        if (recordCount + header.qdcount) {
            ydRunPluginRegex(flow, payload, 0, NULL,
                              (recordCount + header.qdcount), direction,
                              DNS_PORT_NUMBER);
        }
    }
#else /* if defined(YAF_ENABLE_DNSAUTH) && defined(YAF_ENABLE_DNSNXDOMAIN) */
    if (header.qr && !(header.rcode)) {
        if (recordCount) {
            ydRunPluginRegex(flow, payload, 0, NULL, recordCount, direction,
                              DNS_PORT_NUMBER);
        }
    } else {
        if (recordCount + header.qdcount) {
            ydRunPluginRegex(flow, payload, 0, NULL,
                              (recordCount + header.qdcount),
                              direction, DNS_PORT_NUMBER);
        }
    }
#endif /* if defined(YAF_ENABLE_DNSAUTH) && defined(YAF_ENABLE_DNSNXDOMAIN) */
#endif /* if YAF_ENABLE_DPI */

#endif /* ifdef PAYLOAD_INSPECTION */

    /* this is the DNS port code */
    /* fprintf(stderr, " <dns exit 11 match> ");*/
    return DNS_PORT_NUMBER;
}


#if YAF_ENABLE_DPI
void *
ydpProcessDPI(
    ypDPIFlowCtx_t       *flowContext,
    fbSubTemplateList_t  *stl,
    yfFlow_t             *flow,
    uint8_t               fwdcap,
    uint8_t               totalcap)
{
    yfDPIData_t   *dpi         = flowContext->dpi;
    dns_flow_t    *dnsRecord   = NULL;
    uint8_t        recCountFwd = 0;
    uint8_t        recCountRev = 0;
    unsigned int   buflen      = 0;
    int            loop;

    flowContext->exbuf = g_slice_alloc0(flowContext->yfctx->dpi_total_limit);

    if (!flow->rval.payload) {
        totalcap = fwdcap;
    }

    for (loop = flowContext->startOffset; loop < totalcap; ++loop) {
        if (dpi[loop].dpacketID == 0) {
            recCountFwd += dpi[loop].dpacketCapt;
        } else if (dpi[loop].dpacketID == 1) {
            recCountRev += dpi[loop].dpacketCapt;
        }
    }

    dnsRecord = (dns_flow_t *)fbSubTemplateListInit(
        stl, 3, YAF_DNS_FLOW_TID, dns_template, recCountFwd + recCountRev);
    if (!dnsRecord) {
        g_debug("Error initializing SubTemplateList for DNS Resource "
                "Record with %d Templates", recCountFwd + recCountRev);
        return NULL;
    }

    if (flow->val.payload && recCountFwd) {
        ypDNSParser(&dnsRecord, flow, &(flow->val),
                    flowContext->exbuf, &buflen, recCountFwd,
                    flowContext->yfctx->dpi_total_limit);
    }

    if (recCountRev) {
        if (recCountFwd) {
            if (!(dnsRecord = fbSubTemplateListGetNextPtr(stl, dnsRecord))) {
                return (void *)stl;
            }
        }
        if (!flow->rval.payload) {
            /* Uniflow */
            ypDNSParser(&dnsRecord, flow, &(flow->val),
                        flowContext->exbuf, &buflen, recCountRev,
                        flowContext->yfctx->dpi_total_limit);
        } else {
            ypDNSParser(&dnsRecord, flow, &(flow->rval),
                        flowContext->exbuf, &buflen, recCountRev,
                        flowContext->yfctx->dpi_total_limit);
        }
    }

    return (void *)stl;
}

gboolean ydpAddTemplates(
    fbSession_t  *session)
{
    fbTemplateInfo_t *mdInfo;

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_DNS_FLOW_NAME, YAF_DNS_FLOW_DESC, 53,
        FB_TMPL_MD_LEVEL_1);

    if (!ydInitTemplate(&dns_template, session, dns_spec,
                        mdInfo, YAF_DNS_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_DNS_A_FLOW_NAME, YAF_DNS_A_FLOW_DESC, 53,
        YAF_DNS_FLOW_TID);

    if (!ydInitTemplate(&dns_a_template, session, dns_a_spec,
                        mdInfo, YAF_DNS_A_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_DNS_AAAA_FLOW_NAME, YAF_DNS_AAAA_FLOW_DESC, 53,
        YAF_DNS_FLOW_TID);

    if (!ydInitTemplate(&dns_aaaa_template, session, dns_aaaa_spec,
                        mdInfo, YAF_DNS_AAAA_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_DNS_CNAME_FLOW_NAME, YAF_DNS_CNAME_FLOW_DESC, 53,
        YAF_DNS_FLOW_TID);

    if (!ydInitTemplate(&dns_cname_template, session, dns_cname_spec,
                        mdInfo, YAF_DNS_CNAME_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_DNS_MX_FLOW_NAME, YAF_DNS_MX_FLOW_DESC, 53,
        YAF_DNS_FLOW_TID);

    if (!ydInitTemplate(&dns_mx_template, session, dns_mx_spec,
                        mdInfo, YAF_DNS_MX_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_DNS_NS_FLOW_NAME, YAF_DNS_NS_FLOW_DESC, 53,
        YAF_DNS_FLOW_TID);

    if (!ydInitTemplate(&dns_ns_template, session, dns_ns_spec,
                        mdInfo, YAF_DNS_NS_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_DNS_PTR_FLOW_NAME, YAF_DNS_PTR_FLOW_DESC, 53,
        YAF_DNS_FLOW_TID);

    if (!ydInitTemplate(&dns_ptr_template, session, dns_ptr_spec,
                        mdInfo, YAF_DNS_PTR_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_DNS_TXT_FLOW_NAME, YAF_DNS_TXT_FLOW_DESC, 53,
        YAF_DNS_FLOW_TID);

    if (!ydInitTemplate(&dns_txt_template, session, dns_txt_spec,
                        mdInfo, YAF_DNS_TXT_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_DNS_SOA_FLOW_NAME, YAF_DNS_SOA_FLOW_DESC, 53,
        YAF_DNS_FLOW_TID);

    if (!ydInitTemplate(&dns_soa_template, session, dns_soa_spec,
                        mdInfo, YAF_DNS_SOA_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_DNS_SRV_FLOW_NAME, YAF_DNS_SRV_FLOW_DESC, 53,
        YAF_DNS_FLOW_TID);

    if (!ydInitTemplate(&dns_srv_template, session, dns_srv_spec,
                        mdInfo, YAF_DNS_SRV_FLOW_TID, 0))
    {
        return FALSE;
    }

    if (dnssec_global) {
        mdInfo = fbTemplateInfoAlloc();
        fbTemplateInfoInit(
            mdInfo, YAF_DNS_DS_FLOW_NAME, YAF_DNS_DS_FLOW_DESC, 53,
            YAF_DNS_FLOW_TID);

        if (!ydInitTemplate(&dns_ds_template, session, dns_ds_spec,
                            mdInfo, YAF_DNS_DS_FLOW_TID, 0))
        {
            return FALSE;
        }

        mdInfo = fbTemplateInfoAlloc();
        fbTemplateInfoInit(
            mdInfo, YAF_DNS_RRSIG_FLOW_NAME, YAF_DNS_RRSIG_FLOW_DESC, 53,
            YAF_DNS_FLOW_TID);

        if (!ydInitTemplate(&dns_rrsig_template, session, dns_rrsig_spec,
                            mdInfo, YAF_DNS_RRSIG_FLOW_TID, 0))
        {
            return FALSE;
        }

        mdInfo = fbTemplateInfoAlloc();
        fbTemplateInfoInit(
            mdInfo, YAF_DNS_NSEC_FLOW_NAME, YAF_DNS_NSEC_FLOW_DESC, 53,
            YAF_DNS_FLOW_TID);

        if (!ydInitTemplate(&dns_nsec_template, session, dns_nsec_spec,
                            mdInfo, YAF_DNS_NSEC_FLOW_TID, 0))
        {
            return FALSE;
        }

        mdInfo = fbTemplateInfoAlloc();
        fbTemplateInfoInit(
            mdInfo, YAF_DNS_NSEC3_FLOW_NAME, YAF_DNS_NSEC3_FLOW_DESC, 53,
            YAF_DNS_FLOW_TID);

        if (!ydInitTemplate(&dns_nsec3_template, session, dns_nsec3_spec,
                            mdInfo, YAF_DNS_NSEC3_FLOW_TID, 0))
        {
            return FALSE;
        }

        mdInfo = fbTemplateInfoAlloc();
        fbTemplateInfoInit(
            mdInfo, YAF_DNS_NSEC3PARAM_FLOW_NAME,
            YAF_DNS_NSEC3PARAM_FLOW_DESC, 53, YAF_DNS_FLOW_TID);

        if (!ydInitTemplate(
                &dns_nsec3param_template, session, dns_nsec3param_spec,
                mdInfo, YAF_DNS_NSEC3PARAM_FLOW_TID, 0))
        {
            return FALSE;
        }

        mdInfo = fbTemplateInfoAlloc();
        fbTemplateInfoInit(
            mdInfo, YAF_DNS_DNSKEY_FLOW_NAME, YAF_DNS_DNSKEY_FLOW_DESC, 53,
            YAF_DNS_FLOW_TID);

        if (!ydInitTemplate(&dns_dnskey_template, session, dns_dnskey_spec,
                            mdInfo, YAF_DNS_DNSKEY_FLOW_TID, 0))
        {
            return FALSE;
        }
    }
    return TRUE;
}

void
ydpFreeRec(
    ypDPIFlowCtx_t  *flowContext)
{
    dns_flow_t          *dns = NULL;
    fbSubTemplateList_t *stl = (fbSubTemplateList_t *)flowContext->rec;

    if (!stl) {
        return;
    }

    while ((dns = fbSubTemplateListGetNextPtr(stl, dns))) {
        fbSubTemplateListClear(&(dns->dnsRRList));
    }
}
#endif  /* YAF_ENABLE_DPI */


#ifdef PAYLOAD_INSPECTION
/**
 * ycDnsScanRebuildHeader
 *
 * This function handles the endianess of the received message and
 * deals with machine alignment issues by not mapping a network
 * octect stream directly into the DNS structure
 *
 * @param payload a network stream capture
 * @param header a pointer to a client allocated dns message
 *        header structure
 *
 *
 */
static void
ycDnsScanRebuildHeader(
    const uint8_t             *payload,
    ycDnsScanMessageHeader_t  *header)
{
    uint16_t    *tempArray = (uint16_t *)header;
    uint16_t     bitmasks = ntohs(*((uint16_t *)(payload + 2)));
    unsigned int loop;

    memcpy(tempArray, payload, sizeof(ycDnsScanMessageHeader_t));
    for (loop = 0; loop < sizeof(ycDnsScanMessageHeader_t) / sizeof(uint16_t);
         loop++)
    {
        *(tempArray + loop) = ntohs(*(tempArray + loop));
    }

    header->qr = bitmasks & 0x8000 ? 1 : 0;
    header->opcode = (bitmasks & 0x7800) >> 11;
    header->aa = bitmasks & 0x0400 ? 1 : 0;
    header->tc = bitmasks & 0x0200 ? 1 : 0;
    header->rd = bitmasks & 0x0100 ? 1 : 0;
    header->ra = bitmasks & 0x0080 ? 1 : 0;
    header->z = bitmasks & 0x0040 ? 1 : 0;
    /* don't think we care about these
     * header->ad = bitmasks & 0x0020 ? 1 : 0;
     * header->cd = bitmasks & 0x0010 ? 1 : 0; */
    header->rcode = bitmasks & 0x000f;
/*
 *  g_debug("header->qr %d", header->qr);
 *  g_debug("header->opcode %d", header->opcode);
 *  g_debug("header->aa %d", header->aa);
 *  g_debug("header->tc %d", header->tc);
 *  g_debug("header->rd %d", header->rd);
 *  g_debug("header->ra %d", header->ra);
 *  g_debug("header->z %d", header->z);
 *  g_debug("header->rcode %d", header->rcode);
 */
}

static
uint16_t
ycDnsScanCheckResourceRecord(
    const uint8_t  *payload,
    unsigned int   *offset,
    unsigned int    payloadSize)
{
    uint16_t nameSize;
    uint16_t rrType;
    uint16_t rrClass;
    uint16_t rdLength;
    gboolean compress_flag = FALSE;

    if (*offset >= payloadSize) {
        return 0;
    }

    nameSize  = *(payload + (*offset));

    while ((0 != nameSize) && (*offset < payloadSize)) {
        if (DNS_NAME_COMPRESSION == (nameSize & DNS_NAME_COMPRESSION)) {
            *offset += sizeof(uint16_t);
            if (!compress_flag) {
                compress_flag = TRUE;
            }
        } else {
            *offset += nameSize + 1;
        }
        if (*offset >= payloadSize) {
            return 0;
        }
        nameSize = *(payload + (*offset));
    }

    if (!compress_flag) {
        *offset += 1;
    }

    if ((*offset + 2) > payloadSize) {
        return 0;
    }

    /* check the type */
#   if HAVE_ALIGNED_ACCESS_REQUIRED
    rrType = ((*(payload + (*offset))) << 8) |
        ((*(payload + (*offset) + 1)) );
    rrType = ntohs(rrType);
#   else
    rrType = ntohs(*(uint16_t *)(payload + (*offset)));
#   endif /* if HAVE_ALIGNED_ACCESS_REQUIRED */
    *offset += sizeof(uint16_t);

    if (rrType == 0) {
        return 0;
    } else if (rrType > 52) {
        if ((rrType < 249) || (rrType > 253)) {
            if ((rrType != 32769) && (rrType != 32768) && (rrType != 99)) {
                return 0;
            }
        }
    }

    if ((*offset + 2) > payloadSize) {
        return 0;
    }

    /* check the class */
#   if HAVE_ALIGNED_ACCESS_REQUIRED
    rrClass = ((*(payload + (*offset))) << 8) |
        ((*(payload + (*offset) + 1)) );
    rrClass = ntohs(rrClass);
#   else
    rrClass = ntohs(*(uint16_t *)(payload + (*offset)));
#   endif /* if HAVE_ALIGNED_ACCESS_REQUIRED */
    *offset += sizeof(uint16_t);
    /* OPT Records use class field as UDP payload size */
    if (rrClass > 4 && rrType != 41) {
        /* rfc 2136 */
        if (rrClass != 254) {
            return 0;
        }
    }
    /* skip past the time to live */
    *offset += sizeof(uint32_t);

    if ((*offset + 2) > payloadSize) {
        return 0;
    }

    /* get the record data length, (so we can skip ahead the right amount) */
#   if HAVE_ALIGNED_ACCESS_REQUIRED
    rdLength = ((*(payload + (*offset))) << 8) |
        ((*(payload + (*offset) + 1)) );
    rdLength = ntohs(rdLength);
#   else
    rdLength = ntohs(*(uint16_t *)(payload + (*offset)));
#   endif /* if HAVE_ALIGNED_ACCESS_REQUIRED */
    *offset += sizeof(uint16_t);

    /* not going to try to parse the data record, what's in there depends on
     * the class and type fields, but the rdlength field always tells us how
     * many bytes are in it */
    *offset += rdLength;

    if (*offset > payloadSize) {
        return 0;
    }/* the record seems intact enough */
    return rrType;
}


#if YAF_ENABLE_DPI
/**
 * ypGetDNSQName
 *
 * Does the DNS Name Compression Pointer Follow Game - returns the
 * length of the name
 *
 */
static uint8_t
ypGetDNSQName(
    uint8_t        *buf,
    uint16_t        bufoffset,
    const uint8_t  *payload,
    unsigned int    payloadSize,
    uint16_t       *offset,
    uint16_t        export_limit)
{
    uint16_t     nameSize;
    uint16_t     toffset = *(offset);
    gboolean     pointer_flag = FALSE;
    unsigned int pointer_depth = 0;
    uint8_t      temp_buf[DNS_MAX_NAME_LENGTH + 1];
    unsigned int temp_buf_size = 0;

    while (toffset < payloadSize) {
        if (0 == *(payload + toffset) ) {
            if (!pointer_flag) {
                *offset += 1;
            }
            temp_buf[temp_buf_size] = '\0';
            toffset = 0;
            break;
        } else if (DNS_NAME_COMPRESSION ==
                   (*(payload + toffset) & DNS_NAME_COMPRESSION))
        {
            if ( ((size_t)toffset + 1) >= payloadSize) {
                /*Incomplete Name Pointer */
                return 0;
            }
            toffset = ntohs(*((uint16_t *)(payload + toffset)));
            toffset = DNS_NAME_OFFSET & toffset;
            pointer_depth += 1;

            if (pointer_depth > DNS_MAX_NAME_LENGTH) {
                /* Too many pointers in DNS name */
                return 0;
            }

            if (!pointer_flag) {
                *offset += sizeof(uint16_t);
                pointer_flag = TRUE;
            }

            continue;
        } else if (0 == (*(payload + toffset) & DNS_NAME_COMPRESSION)) {
            nameSize = *(payload + toffset);
            if ( (nameSize + temp_buf_size + 1) > DNS_MAX_NAME_LENGTH) {
                /* DNS Name Too Long */
                return 0;
            }
            memcpy(temp_buf + temp_buf_size, (payload + toffset + 1),
                   nameSize);
            temp_buf[temp_buf_size + nameSize] = '.';
            temp_buf_size += nameSize + 1;
            if (!pointer_flag) {
                *offset += *(payload + toffset) + 1;
            }

            toffset += nameSize + 1;
        } else if (0x40 == (*(payload + toffset) & DNS_NAME_COMPRESSION)) {
            /* See RFC6891, Extension Mechanisms for DNS (EDNS(0)),
             * which obsoletes RFC2671, RFC2673 */
            /* YAF does not support this */
            /* g_debug("Extended DNS label types (%#04x) are not supported", */
            /*         *(payload + toffset)); */
            return 0;
        } else {
            g_assert(0x80 == (*(payload + toffset) & DNS_NAME_COMPRESSION));
            /* g_debug("Unknown DNS label type %#04x", *(payload + toffset)); */
            return 0;
        }
    }

    if (toffset >= payloadSize) {
        /*DNS Name outside payload */
        return 0;
    }

    if (bufoffset + temp_buf_size > export_limit) {
        /* Name too large to export in allowed buffer size*/
        return 0;
    }

    /* skip trailing '.' */
    memcpy(buf + bufoffset, temp_buf, temp_buf_size);
    bufoffset += temp_buf_size;

    return temp_buf_size;
}


static void
ypDNSParser(
    dns_flow_t    **dnsRecord,
    yfFlow_t       *flow,
    yfFlowVal_t    *val,
    uint8_t        *buf,
    unsigned int   *bufLen,
    uint8_t         recordCount,
    uint16_t        export_limit)
{
    ycDnsScanMessageHeader_t header;
    uint16_t       payloadOffset = sizeof(ycDnsScanMessageHeader_t);
    uint16_t       firstpkt = val->paylen;
    uint16_t       msglen;
    size_t         nameLen;
    uint8_t        nxdomain = 0;
    unsigned int   bufSize = (*bufLen);
    uint16_t       rrType;
    unsigned int   loop = 0;
    const uint8_t *payload = val->payload;
    unsigned int   payloadSize = val->paylen;

    if (flow->key.proto == YF_PROTO_TCP) {
        while (loop < val->pkt && loop < YAF_MAX_PKT_BOUNDARY) {
            if (val->paybounds[loop] == 0) {
                loop++;
            } else {
                firstpkt = val->paybounds[loop];
                break;
            }
        }
        msglen = ntohs(*((uint16_t *)(payload)));
        if ((msglen + 2) == firstpkt) {
            /* this is the weird message length in TCP */
            payload += sizeof(uint16_t);
            payloadSize -= sizeof(uint16_t);
        }
    }

    ycDnsScanRebuildHeader(payload, &header);

    if (header.rcode != 0) {
        nxdomain = 1;
    }

#if defined(YAF_ENABLE_DNSAUTH)
    if (header.aa) {
        /* get the query part if authoritative */
        nxdomain = 1;
    }
#endif /* if defined(YAF_ENABLE_DNSAUTH) */
    for (loop = 0; loop < header.qdcount; loop++) {
        nameLen = ypGetDNSQName(buf, bufSize, payload, payloadSize,
                                &payloadOffset, export_limit);
        if ((!header.qr || nxdomain)) {
            fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_A_FLOW_TID, dns_a_template, 0);
            (*dnsRecord)->dnsName.len = nameLen;
            (*dnsRecord)->dnsName.buf = buf + bufSize;
            bufSize += (*dnsRecord)->dnsName.len;
            (*dnsRecord)->dnsAuthoritative = header.aa;
            (*dnsRecord)->dnsResponseCode = header.rcode;
            (*dnsRecord)->dnsSection = 0;
            (*dnsRecord)->dnsQueryResponse = header.qr;
            (*dnsRecord)->dnsId = header.id;
            if (((size_t)payloadOffset + 2) < payloadSize) {
                (*dnsRecord)->dnsRRType = ntohs(*((uint16_t *)(payload +
                                                                payloadOffset)));
            }

            recordCount--;
            if (recordCount) {
                (*dnsRecord)++;
            } else {
                *bufLen = bufSize;
                return;
            }
        }

        payloadOffset += (sizeof(uint16_t) * 2);
        /* skip over class */
        if (payloadOffset > payloadSize) {
            goto err;
        }
    }

    for (loop = 0; loop < header.ancount; loop++) {
        (*dnsRecord)->dnsSection = 1;
        (*dnsRecord)->dnsAuthoritative = header.aa;
        (*dnsRecord)->dnsResponseCode = header.rcode;
        (*dnsRecord)->dnsQueryResponse = 1;
        (*dnsRecord)->dnsId = header.id;
        rrType = ypDnsScanResourceRecord(dnsRecord, payload, payloadSize,
                                         &payloadOffset, buf, &bufSize,
                                         export_limit);
        if (rrType != 41) {
            recordCount--;
            if (recordCount) {
                (*dnsRecord)++;
            } else {
                *bufLen = bufSize;
                return;
            }
        }

        if (payloadOffset > payloadSize) {
            goto err;
        }

        if (bufSize > export_limit) {
            bufSize = export_limit;
            goto err;
        }
    }

    for (loop = 0; loop < header.nscount; loop++) {
        (*dnsRecord)->dnsSection = 2;
        (*dnsRecord)->dnsAuthoritative = header.aa;
        (*dnsRecord)->dnsResponseCode = header.rcode;
        (*dnsRecord)->dnsQueryResponse = 1;
        (*dnsRecord)->dnsId = header.id;
        rrType = ypDnsScanResourceRecord(dnsRecord, payload, payloadSize,
                                         &payloadOffset, buf, &bufSize,
                                         export_limit);
        if (rrType != 41) {
            recordCount--;
            if (recordCount) {
                (*dnsRecord)++;
            } else {
                *bufLen = bufSize;
                return;
            }
        }

        if (payloadOffset > payloadSize) {
            goto err;
        }

        if (bufSize > export_limit) {
            bufSize = export_limit;
            goto err;
        }
    }

    for (loop = 0; loop < header.arcount; loop++) {
        (*dnsRecord)->dnsSection = 3;
        (*dnsRecord)->dnsAuthoritative = header.aa;
        (*dnsRecord)->dnsResponseCode = header.rcode;
        (*dnsRecord)->dnsQueryResponse = 1;
        (*dnsRecord)->dnsId = header.id;
        rrType = ypDnsScanResourceRecord(dnsRecord, payload, payloadSize,
                                         &payloadOffset, buf, &bufSize,
                                         export_limit);
        if (rrType != 41) {
            recordCount--;
            if (recordCount) {
                (*dnsRecord)++;
            } else {
                *bufLen = bufSize;
                return;
            }
        }

        if (payloadOffset > payloadSize) {
            goto err;
        }

        if (bufSize > export_limit) {
            bufSize = export_limit;
            goto err;
        }
    }

    *bufLen = bufSize;

    return;

  err:
    *bufLen = bufSize;
    /* something went wrong so we need to pad the rest of the STL with NULLs */
    /* Most likely we ran out of space in the DNS Export Buffer */
    while (recordCount) {
        fbSubTemplateListInit(&((*dnsRecord)->dnsRRList), 3,
                              YAF_DNS_A_FLOW_TID, dns_a_template, 0);
        recordCount--;
        if (recordCount) {(*dnsRecord)++;}
    }
}


static uint16_t
ypDnsScanResourceRecord(
    dns_flow_t    **dnsRecord,
    const uint8_t  *payload,
    unsigned int    payloadSize,
    uint16_t       *offset,
    uint8_t        *buf,
    unsigned int   *bufLen,
    uint16_t        export_limit)
{
    uint16_t rrLen = 0;
    uint16_t rrType;
    uint16_t temp_offset;
    uint16_t bufSize = *bufLen;

    (*dnsRecord)->dnsName.len = ypGetDNSQName(
        buf, bufSize, payload, payloadSize, offset, export_limit);
    (*dnsRecord)->dnsName.buf = buf + bufSize;
    bufSize += (*dnsRecord)->dnsName.len;

    /* rrType */
    READ_U16_INC(&((*dnsRecord)->dnsRRType), offset, payload, payloadSize);
    rrType = (*dnsRecord)->dnsRRType;
    /* ignore class */
    *offset += sizeof(uint16_t);
    /* time to live */
    READ_U32_INC(&((*dnsRecord)->dnsTTL), offset, payload, payloadSize);
    /* length */
    READ_U16_INC(&rrLen, offset, payload, payloadSize);

    if (*offset >= payloadSize) {
        fbSubTemplateListInit(&((*dnsRecord)->dnsRRList), 3,
                              YAF_DNS_A_FLOW_TID, dns_a_template, 0);
        return rrType;
    }

    temp_offset = (*offset);

    switch (rrType) {

      case 1:
        /* A */
        {
            dns_a_flow_t *arecord = (dns_a_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_A_FLOW_TID, dns_a_template, 1);
            READ_U32_INC(&arecord->dnsA, &temp_offset, payload, payloadSize);
        }
        break;

      case 2:
        /* NS */
        {
            dns_ns_flow_t *nsrecord = (dns_ns_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_NS_FLOW_TID, dns_ns_template, 1);
            nsrecord->dnsNSDName.len = ypGetDNSQName(
                buf, bufSize, payload, payloadSize, &temp_offset, export_limit);
            nsrecord->dnsNSDName.buf = buf + bufSize;
            bufSize += nsrecord->dnsNSDName.len;
        }
        break;

      case 5:
        /* CNAME */
        {
            dns_cname_flow_t *cname = (dns_cname_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_CNAME_FLOW_TID, dns_cname_template, 1);
            cname->dnsCNAME.len = ypGetDNSQName(
                buf, bufSize, payload, payloadSize, &temp_offset, export_limit);
            cname->dnsCNAME.buf = buf + bufSize;
            bufSize += cname->dnsCNAME.len;
        }
        break;

      case 12:
        /* PTR */
        {
            dns_ptr_flow_t *ptr = (dns_ptr_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_PTR_FLOW_TID, dns_ptr_template, 1);
            ptr->dnsPTRDName.len = ypGetDNSQName(
                buf, bufSize, payload, payloadSize, &temp_offset, export_limit);
            ptr->dnsPTRDName.buf = buf + bufSize;
            bufSize += ptr->dnsPTRDName.len;
        }
        break;

      case 15:
        /* MX */
        {
            dns_mx_flow_t *mx = (dns_mx_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_MX_FLOW_TID, dns_mx_template, 1);
            READ_U16_INC(&mx->dnsMXPreference, &temp_offset,
                         payload, payloadSize);
            mx->dnsMXExchange.len = ypGetDNSQName(
                buf, bufSize, payload, payloadSize, &temp_offset, export_limit);
            mx->dnsMXExchange.buf = buf + bufSize;
            bufSize += mx->dnsMXExchange.len;
        }
        break;

      case 16:
        /* TXT */
        {
            dns_txt_flow_t *txt = (dns_txt_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_TXT_FLOW_TID, dns_txt_template, 1);
            READ_U8_INC(&txt->dnsTXTData.len, &temp_offset,
                        payload, payloadSize);
            if (txt->dnsTXTData.len + temp_offset > payloadSize) {
                txt->dnsTXTData.len = 0;
                break;
            }
            txt->dnsTXTData.buf = (uint8_t *)payload + temp_offset;
            /* not storing in exbuf, but it counts against export_limit */
            bufSize += txt->dnsTXTData.len;
        }
        break;

      case 28:
        /* AAAA */
        {
            dns_aaaa_flow_t *aa = (dns_aaaa_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_AAAA_FLOW_TID, dns_aaaa_template, 1);
            memcpy(aa->dnsAAAA, (payload + temp_offset), sizeof(aa->dnsAAAA));
        }
        break;

      case 6:
        /* SOA */
        {
            dns_soa_flow_t *soa = (dns_soa_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_SOA_FLOW_TID, dns_soa_template, 1);
            soa->dnsSOAMName.len = ypGetDNSQName(
                buf, bufSize, payload, payloadSize, &temp_offset, export_limit);
            soa->dnsSOAMName.buf = buf + bufSize;
            bufSize += soa->dnsSOAMName.len;
            if (temp_offset >= payloadSize) {
                break;
            }
            soa->dnsSOARName.len = ypGetDNSQName(
                buf, bufSize, payload, payloadSize, &temp_offset, export_limit);
            soa->dnsSOARName.buf = buf + bufSize;
            bufSize += soa->dnsSOARName.len;
            if (temp_offset >= payloadSize) {
                break;
            }
            READ_U32_INC(&soa->dnsSOASerial, &temp_offset,
                         payload, payloadSize);
            READ_U32_INC(&soa->dnsSOARefresh, &temp_offset,
                         payload, payloadSize);
            READ_U32_INC(&soa->dnsSOARetry, &temp_offset,
                         payload, payloadSize);
            READ_U32_INC(&soa->dnsSOAExpire, &temp_offset,
                         payload, payloadSize);
            READ_U32_INC(&soa->dnsSOAMinimum, &temp_offset,
                         payload, payloadSize);
        }
        break;

      case 33:
        /* SRV */
        {
            dns_srv_flow_t *srv = (dns_srv_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_SRV_FLOW_TID, dns_srv_template, 1);
            READ_U16_INC(&srv->dnsSRVPriority, &temp_offset,
                         payload, payloadSize);
            READ_U16_INC(&srv->dnsSRVWeight, &temp_offset,
                         payload, payloadSize);
            READ_U16_INC(&srv->dnsSRVPort, &temp_offset,
                         payload, payloadSize);
            srv->dnsSRVTarget.len = ypGetDNSQName(
                buf, bufSize, payload, payloadSize, &temp_offset, export_limit);
            srv->dnsSRVTarget.buf = buf + bufSize;
            bufSize += srv->dnsSRVTarget.len;
        }
        break;

      case 43:
        /* DS */
        if (!dnssec_global) {
            fbSubTemplateListInit(&((*dnsRecord)->dnsRRList), 3,
                                  YAF_DNS_A_FLOW_TID, dns_a_template, 0);
        } else {
            dns_ds_flow_t *ds = (dns_ds_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_DS_FLOW_TID, dns_ds_template, 1);
            READ_U16_INC(&ds->dnsDSKeyTag, &temp_offset,
                         payload, payloadSize);
            READ_U8_INC(&ds->dnsDSAlgorithm, &temp_offset,
                        payload, payloadSize);
            READ_U8_INC(&ds->dnsDSDigestType, &temp_offset,
                        payload, payloadSize);

            /* length of rrdata is rrLen - we know these 3 fields */
            /* should add up to 4 - so rest is digest */
            if ((size_t)rrLen + *offset > payloadSize) {
                break;
            }
            ds->dnsDSDigest.buf = (uint8_t *)payload + temp_offset;
            ds->dnsDSDigest.len = rrLen - 4;
            /* not storing in exbuf, but it counts against export_limit */
            bufSize += ds->dnsDSDigest.len;
        }
        break;

      case 46:
        /* RRSIG */
        if (!dnssec_global) {
            fbSubTemplateListInit(&((*dnsRecord)->dnsRRList), 3,
                                  YAF_DNS_A_FLOW_TID, dns_a_template, 0);
        } else {
            dns_rrsig_flow_t *rrsig = (dns_rrsig_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_RRSIG_FLOW_TID, dns_rrsig_template, 1);

            READ_U16_INC(&rrsig->dnsRRSIGTypeCovered, &temp_offset,
                         payload, payloadSize);
            READ_U8_INC(&rrsig->dnsRRSIGAlgorithm, &temp_offset,
                        payload, payloadSize);
            READ_U8_INC(&rrsig->dnsRRSIGLabels, &temp_offset,
                        payload, payloadSize);
            READ_U32_INC(&rrsig->dnsRRSIGOriginalTTL, &temp_offset,
                         payload, payloadSize);
            READ_U32_INC(&rrsig->dnsRRSIGSignatureExpiration, &temp_offset,
                         payload, payloadSize);
            READ_U32_INC(&rrsig->dnsRRSIGSignatureInception, &temp_offset,
                         payload, payloadSize);
            READ_U16_INC(&rrsig->dnsRRSIGKeyTag, &temp_offset,
                         payload, payloadSize);

            rrsig->dnsRRSIGSigner.len = ypGetDNSQName(
                buf, bufSize, payload, payloadSize, &temp_offset, export_limit);
            rrsig->dnsRRSIGSigner.buf = buf + bufSize;
            bufSize += rrsig->dnsRRSIGSigner.len;

            /* we are (temp_offset - *offset) into the rrdata; the remainder
             * is the signature */
            if ((size_t)rrLen + *offset > payloadSize) {
                break;
            }
            rrsig->dnsRRSIGSignature.buf = (uint8_t *)payload + temp_offset;
            rrsig->dnsRRSIGSignature.len = rrLen - (temp_offset - *offset);
            bufSize += rrsig->dnsRRSIGSignature.len;
        }
        break;

      case 47:
        /* NSEC */
        if (!dnssec_global) {
            fbSubTemplateListInit(&((*dnsRecord)->dnsRRList), 3,
                                  YAF_DNS_A_FLOW_TID, dns_a_template, 0);
        } else {
            dns_nsec_flow_t *nsec = (dns_nsec_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_NSEC_FLOW_TID, dns_nsec_template, 1);

            nsec->dnsNSECNextDomainName.len = ypGetDNSQName(
                buf, bufSize, payload, payloadSize, &temp_offset, export_limit);
            nsec->dnsNSECNextDomainName.buf = buf + bufSize;
            bufSize += nsec->dnsNSECNextDomainName.len;

            /* we are (temp_offset - *offset) into the rrdata; the remainder
             * is the TypeBitMaps */
            if ((size_t)rrLen + *offset > payloadSize) {
                break;
            }
            nsec->dnsNSECTypeBitMaps.buf = (uint8_t *)payload + temp_offset;
            nsec->dnsNSECTypeBitMaps.len = rrLen - (temp_offset - *offset);
            bufSize += nsec->dnsNSECTypeBitMaps.len;
        }
        break;

      case 48:
        /* DNSKEY */
        if (!dnssec_global) {
            fbSubTemplateListInit(&((*dnsRecord)->dnsRRList), 0,
                                  YAF_DNS_A_FLOW_TID, dns_a_template, 0);
        } else {
            dns_dnskey_flow_t *dnskey;
            dnskey = (dns_dnskey_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_DNSKEY_FLOW_TID, dns_dnskey_template, 1);

            READ_U16_INC(&dnskey->dnsDNSKEYFlags, &temp_offset,
                         payload, payloadSize);
            READ_U8_INC(&dnskey->dnsDNSKEYProtocol, &temp_offset,
                        payload, payloadSize);
            READ_U8_INC(&dnskey->dnsDNSKEYAlgorithm, &temp_offset,
                        payload, payloadSize);

            if ((size_t)rrLen + *offset > payloadSize) {
                break;
            }
            dnskey->dnsDNSKEYPublicKey.buf = (uint8_t *)payload + temp_offset;
            dnskey->dnsDNSKEYPublicKey.len = rrLen - 4;
            bufSize += dnskey->dnsDNSKEYPublicKey.len;
        }
        break;

      case 50:
        /* NSEC3 */
        if (!dnssec_global) {
            fbSubTemplateListInit(&((*dnsRecord)->dnsRRList), 0,
                                  YAF_DNS_A_FLOW_TID, dns_a_template, 0);
        } else {
            dns_nsec3_flow_t *nsec3 = (dns_nsec3_flow_t *)fbSubTemplateListInit(
                &((*dnsRecord)->dnsRRList), 3,
                YAF_DNS_NSEC3_FLOW_TID, dns_nsec3_template, 1);

            READ_U8_INC(&nsec3->dnsNSEC3Algorithm, &temp_offset,
                        payload, payloadSize);
            READ_U8_INC(&nsec3->dnsNSEC3Flags, &temp_offset,
                        payload, payloadSize);
            READ_U16_INC(&nsec3->dnsNSEC3Iterations, &temp_offset,
                         payload, payloadSize);

            READ_U8_INC(&nsec3->dnsNSEC3Salt.len, &temp_offset,
                        payload, payloadSize);
            if (nsec3->dnsNSEC3Salt.len + temp_offset > payloadSize) {
                nsec3->dnsNSEC3Salt.len = 0;
                break;
            }
            nsec3->dnsNSEC3Salt.buf = (uint8_t *)payload + temp_offset;
            bufSize += nsec3->dnsNSEC3Salt.len;
            if (bufSize > export_limit) {
                break;
            }
            temp_offset += nsec3->dnsNSEC3Salt.len;

            READ_U8_INC(&nsec3->dnsNSEC3NextHashedOwnerName.len,
                        &temp_offset, payload, payloadSize);
            if (nsec3->dnsNSEC3NextHashedOwnerName.len + temp_offset >
                payloadSize)
            {
                nsec3->dnsNSEC3NextHashedOwnerName.len = 0;
                break;
            }
            nsec3->dnsNSEC3NextHashedOwnerName.buf =
                (uint8_t *)payload + temp_offset;
            bufSize += nsec3->dnsNSEC3NextHashedOwnerName.len;
            if (bufSize > export_limit) {
                break;
            }
            temp_offset += nsec3->dnsNSEC3NextHashedOwnerName.len;

            /* we have moved (temp_offset - *offset) bytes into the record;
             * subtract that from rrLen to get length of TypeBitMaps */
            if ((size_t)rrLen + *offset > payloadSize) {
                break;
            }
            nsec3->dnsNSEC3TypeBitMaps.buf = (uint8_t *)payload + temp_offset;
            nsec3->dnsNSEC3TypeBitMaps.len = rrLen - (temp_offset - *offset);
            bufSize += nsec3->dnsNSEC3TypeBitMaps.len;
        }
        break;

      case 51:
        /* NSEC3PARAM */
        if (!dnssec_global) {
            fbSubTemplateListInit(&((*dnsRecord)->dnsRRList), 0,
                                  YAF_DNS_A_FLOW_TID, dns_a_template, 0);
        } else {
            dns_nsec3param_flow_t *nsec3param =
                (dns_nsec3param_flow_t *)fbSubTemplateListInit(
                    &((*dnsRecord)->dnsRRList), 3,
                    YAF_DNS_NSEC3PARAM_FLOW_TID, dns_nsec3param_template, 1);

            READ_U8_INC(&nsec3param->dnsNSEC3PARAMAlgorithm, &temp_offset,
                        payload, payloadSize);
            READ_U8_INC(&nsec3param->dnsNSEC3PARAMFlags, &temp_offset,
                        payload, payloadSize);
            READ_U16_INC(&nsec3param->dnsNSEC3PARAMIterations, &temp_offset,
                         payload, payloadSize);

            READ_U8_INC(&nsec3param->dnsNSEC3PARAMSalt.len, &temp_offset,
                        payload, payloadSize);
            if (nsec3param->dnsNSEC3PARAMSalt.len + temp_offset > payloadSize) {
                nsec3param->dnsNSEC3PARAMSalt.len = 0;
                break;
            }
            nsec3param->dnsNSEC3PARAMSalt.buf
                = (uint8_t *)payload + temp_offset;
            bufSize += nsec3param->dnsNSEC3PARAMSalt.len;
        }
        break;
      default:
        fbSubTemplateListInit(&((*dnsRecord)->dnsRRList), 3,
                              YAF_DNS_A_FLOW_TID, dns_a_template, 0);
        break;
    }

    *offset += rrLen;
    *bufLen = bufSize;
    return rrType;
}
#endif  /* YAF_ENABLE_DPI */
#endif /* ifdef PAYLOAD_INSPECTION */


/**
 * ydpInitialize
 *
 * Processes the plugin's arguments to determine whether to enable dnssec
 * export.
 *
 */
int
ydpInitialize(
    int             argc,
    char           *argv[],
    uint16_t        applabel,
    gboolean        applabelOnly,
    GError        **err)
{
#if YAF_ENABLE_DPI
    /* first argument determines whether to enable dnssec export */
    if (!applabelOnly && argc >= 2) {
        if (0 == strcmp("true", argv[1])) {
            dnssec_global = TRUE;
            g_debug("DNSSEC export enabled.");
        }
    }
#endif

    return 1;
}
