/*
 *  Copyright 2007-2022 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/**
 *  @internal
 *
 *  @file ldpplugin.c
 *
 *
 *  This recognizes Label Distribution Protocol (LDP) Packets
 *  see http://www.ietf.org/rfc/rfc3036 for more info
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 3.0.0
 *
 *  Copyright 2022 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Released under a GNU GPL 2.0-style license, please see license.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
 *  Trademark Office by Carnegie Mellon University.
 *
 *  This Software includes and/or makes use of Third-Party Software subject
 *  to its own license.
 *
 *  DM22-0007
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <yaf/yafDPIPlugin.h>


#define LDP_PORT_NUMBER  646
#define LDP_VERSION 1

/**
 * ydpScanPayload
 *
 * the scanner for recognizing LDP packets
 *
 * @param payload the packet payload
 * @param payloadSize size of the packet payload
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 *
 * @return LDP_PORT_NUMBER for LDP packets,
 *         otherwise 0
 */
uint16_t
ydpScanPayload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    unsigned int offsetptr = 0;
    uint16_t     version;
    uint16_t     length;
    uint32_t     id;

    /* Only do decode if MPLS is enabled */
#ifndef YAF_MPLS
    return 0;
#endif

    /* BGP header is fixed - has to be at least 10 */
    if (payloadSize < 10) {
        return 0;
    }

    version = g_ntohs(*(uint16_t *)payload);
    if (version != LDP_VERSION) {
        return 0;
    }

    offsetptr += 2;

    length = g_ntohs(*(uint16_t *)(payload + offsetptr));

    if (length > 4096) {
        return 0;
    }

    if (length < 6) {
        return 0;
    }

    offsetptr += 2;

    id = g_ntohl(*(uint32_t *)(payload + offsetptr));

    /* id should be the same as src ip */

    if (id != flow->key.addr.v4.sip) {
        return 0;
    }

    offsetptr += 4;

    /* Last 2 bytes are 0 */

    if (*(payload + offsetptr) != 0) {
        return 0;
    }

    offsetptr++;

    if (*(payload + offsetptr) != 0) {
        return 0;
    }

    return LDP_PORT_NUMBER;
}
