/*
 *  Copyright 2007-2022 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/**
 *  @internal
 *
 *  @file nntpplugin.c
 *
 *  this provides NNTP payload packet recognition for use within YAF
 *  It is based on RFC 977 and some random limited packet capture.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Ecoff
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 3.0.0
 *
 *  Copyright 2022 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Released under a GNU GPL 2.0-style license, please see license.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
 *  Trademark Office by Carnegie Mellon University.
 *
 *  This Software includes and/or makes use of Third-Party Software subject
 *  to its own license.
 *
 *  DM22-0007
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>

#include <yaf/yafDPIPlugin.h>
#if YAF_ENABLE_DPI

#define YAF_NNTP_FLOW_TID   0xCD00
#define YAF_NNTP_FLOW_NAME  "yaf_nntp"
#define YAF_NNTP_FLOW_DESC  NULL

static fbInfoElementSpec_t yaf_nntp_spec[] = {
    {"nntpResponseList",    FB_IE_VARLEN, 0 },
    {"nntpCommandList",     FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

typedef struct yfNNTPFlow_st {
    fbBasicList_t   nntpResponse;
    fbBasicList_t   nntpCommand;
} yfNNTPFlow_t;

static fbTemplate_t     *nntpTemplate;
#endif

#include <pcre.h>


#define NNTP_PORT 119

/**
 * the compiled regular expressions, and related
 * flags
 *
 */
static pcre        *nntpCommandRegex = NULL;
static pcre        *nntpResponseRegex = NULL;


/*static int ycDebugBinPrintf(uint8_t *data, uint16_t size);*/


void
nntpFillBasicList(
    yfFlow_t      *flow,
    yfDPIData_t   *dpi,
    uint8_t        totalCaptures,
    uint8_t        forwardCaptures,
    fbVarfield_t **varField,
    uint8_t       *indexArray)
{
    yfFlowVal_t *val;
    unsigned int i;

    if (!(*varField)) {
        return;
    }
    for (i = 0; i < totalCaptures; i++) {
        val = (indexArray[i] < forwardCaptures) ? &flow->val : &flow->rval;
        if (dpi[indexArray[i]].dpacketCapt + dpi[indexArray[i]].dpacketCaptLen
            > val->paylen)
        {
            continue;
        }
        if (val->payload) {
            (*varField)->buf = val->payload + dpi[indexArray[i]].dpacketCapt;
            (*varField)->len = dpi[indexArray[i]].dpacketCaptLen;
        }
        if (i + 1 < totalCaptures) {
            (*varField)++;
        }
    }
}


/**
 * ydpScanPayload
 *
 * scans a given payload to see if it conforms to our idea of what NNTP traffic
 * looks like.
 *
 *
 *
 * @param payload pointer to the payload data
 * @param payloadSize the size of the payload parameter
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 * @return 0 for no match NNTP_PORT_NUMBER (119) for a match
 *
 */
uint16_t
ydpScanPayload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    int rc;
#   define NUM_CAPT_VECTS 60
    int vects[NUM_CAPT_VECTS];

    rc = pcre_exec(nntpCommandRegex, NULL, (char *)payload, payloadSize,
                   0, 0, vects, NUM_CAPT_VECTS);

    if (rc <= 0) {
        rc = pcre_exec(nntpResponseRegex, NULL, (char *)payload,
                       payloadSize, 0, 0, vects, NUM_CAPT_VECTS);
    }

    /** at some point in the future, this is the place to extract protocol
     *  information like message targets and join targets, etc.*/
#if YAF_ENABLE_DPI
    if (rc > 0) {
        ydRunPluginRegex(flow, payload, payloadSize, nntpCommandRegex, 0,
                          173, NNTP_PORT);
        ydRunPluginRegex(flow, payload, payloadSize, nntpResponseRegex, 0,
                          172, NNTP_PORT);
    }
#endif /* if YAF_ENABLE_DPI */

    if (rc > 0) {
        return NNTP_PORT;
    }

    return 0;
}

#if YAF_ENABLE_DPI
void *
ydpProcessDPI(
    ypDPIFlowCtx_t       *flowContext,
    fbSubTemplateList_t  *stl,
    yfFlow_t             *flow,
    uint8_t               fwdcap,
    uint8_t               totalcap)
{
    yfDPIData_t   *dpi = flowContext->dpi;
    yfNNTPFlow_t  *rec = NULL;
    fbInfoModel_t *model = ydGetDPIInfoModel();
    uint8_t        count;
    uint8_t        start = flowContext->startOffset;
    int            total = 0;
    fbVarfield_t  *nntpVar = NULL;
    uint8_t        totalIndex[YAF_MAX_CAPTURE_FIELDS];
    const fbInfoElement_t *nntpResponse;
    const fbInfoElement_t *nntpCommand;

    rec = (yfNNTPFlow_t *)fbSubTemplateListInit(stl, 3, YAF_NNTP_FLOW_TID,
                                                nntpTemplate, 1);
    if (!flow->rval.payload) {
        totalcap = fwdcap;
    }

    /* nntp Response */
    for (count = start; count < totalcap; count++) {
        if (dpi[count].dpacketID == 172) {
            totalIndex[total] = count;
            total++;
        }
    }

    nntpResponse = fbInfoModelGetElementByName(model, "nntpResponse");
    nntpVar = (fbVarfield_t *)fbBasicListInit(
        &(rec->nntpResponse), 3, nntpResponse, total);

    nntpFillBasicList(flow, dpi, total, fwdcap, &nntpVar, totalIndex);

    total = 0;
    nntpVar = NULL;
    /* nntp Command */
    for (count = start; count < totalcap; count++) {
        if (dpi[count].dpacketID == 173) {
            totalIndex[total] = count;
            total++;
        }
    }

    nntpCommand = fbInfoModelGetElementByName(model, "nntpCommand");
    nntpVar = (fbVarfield_t *)fbBasicListInit(
        &(rec->nntpCommand), 3, nntpCommand, total);

    nntpFillBasicList(flow, dpi, total, fwdcap, &nntpVar, totalIndex);

    return (void *)rec;
}

gboolean ydpAddTemplates(
    fbSession_t  *session)
{
    fbTemplateInfo_t *mdInfo;
    const fbInfoElement_t *bl_element;

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_NNTP_FLOW_NAME, YAF_NNTP_FLOW_DESC, 119,
        FB_TMPL_MD_LEVEL_1);

    /* ruleset does not contain IE information, add metadata manually */
    bl_element = ydLookupNamedBlByID(CERT_PEN, 172);
    if (bl_element) {
        fbTemplateInfoAddBasicList(mdInfo, bl_element->ent, bl_element->num,
                                   CERT_PEN, 172);
    }
    bl_element = ydLookupNamedBlByID(CERT_PEN, 173);
    if (bl_element) {
        fbTemplateInfoAddBasicList(mdInfo, bl_element->ent, bl_element->num,
                                   CERT_PEN, 173);
    }

    if (!ydInitTemplate(&nntpTemplate, session, yaf_nntp_spec,
                        mdInfo, YAF_NNTP_FLOW_TID, 0))
    {
        return FALSE;
    }
    return TRUE;
}

void ydpFreeRec(
    ypDPIFlowCtx_t  *flowContext)
{
    yfNNTPFlow_t *rec = (yfNNTPFlow_t *)flowContext->rec;

    fbBasicListClear(&(rec->nntpResponse));
    fbBasicListClear(&(rec->nntpCommand));
}
#endif  /* YAF_ENABLE_DPI */

/**
 * ydpInitialize
 *
 * this initializes the PCRE expressions needed to search the payload for
 * NNTP
 *
 *
 * @sideeffect sets the initialized flag on success
 *
 * @return 1 if initialization is complete correctly, 0 otherwise
 */
int
ydpInitialize(
    int             argc,
    char           *argv[],
    uint16_t        applabel,
    gboolean        applabelOnly,
    GError        **err)
{
    const char  nntpResponseRegexString[] =
        "(([1-5]([0-4]|[8-9])[0-9] )+"
        ".* (text follows)?[\\r\\n]?"
        "(.* \\r\\n)?)\b";
    const char nntpCommandRegexString[] =
        "^((ARTICLE|GROUP|HELP|IHAVE|LAST"
        "|LIST|NEWGROUPS|NEWNEWS|NEXT|POST|QUIT"
        "|SLAVE|STAT|MODE) ?[ a-zA-Z0-9.]*)[ \\r\\n]";

    nntpCommandRegex = ydPcreCompile(nntpCommandRegexString, 0);
    nntpResponseRegex = ydPcreCompile(nntpResponseRegexString,
                                      PCRE_EXTENDED | PCRE_ANCHORED);

    return (NULL != nntpCommandRegex && NULL != nntpResponseRegex);
}


#if 0
static int
ycDebugBinPrintf(
    uint8_t   *data,
    uint16_t   size)
{
    uint16_t loop;
    int      numPrinted = 0;

    for (loop = 0; loop < size; loop++) {
        if (isprint(*(data + loop)) && !iscntrl(*(data + loop))) {
            printf("%c", *(data + loop));
        } else {
            printf(".");
        }
        if ('\n' == *(data + loop) || '\r' == *(data + loop)
            || '\0' == (data + loop))
        {
            break;
        }
        numPrinted++;
    }

    return numPrinted;
}
#endif /* 0 */
