/*
 *  Copyright 2007-2022 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/**
 *  @internal
 *
 *  @file ircplugin.c
 *
 *  this provides IRC payload packet recognition for use within YAF
 *  It is based on RFC 2812 and some random limited packet capture.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Chris Inacio
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 3.0.0
 *
 *  Copyright 2022 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Released under a GNU GPL 2.0-style license, please see license.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
 *  Trademark Office by Carnegie Mellon University.
 *
 *  This Software includes and/or makes use of Third-Party Software subject
 *  to its own license.
 *
 *  DM22-0007
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <pcre.h>

#include <yaf/yafDPIPlugin.h>
#if YAF_ENABLE_DPI

#define YAF_IRC_FLOW_TID    0xC200
#define YAF_IRC_FLOW_NAME   "yaf_irc"
#define YAF_IRC_FLOW_DESC   NULL

static fbInfoElementSpec_t yaf_irc_spec[] = {
    {"ircTextMessageList",      FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

typedef struct yfIRCFlow_st {
    fbBasicList_t   ircMsg;
} yfIRCFlow_t;

static fbTemplate_t     *ircTemplate;
#endif  /* YAF_ENABLE_DPI */

#define IRCDEBUG 0
#define IRC_PORT 194

/**
 * the compiled regular expressions, and related
 * flags
 *
 */
static pcre        *ircMsgRegex = NULL;
/*static pcre *ircJoinRegex = NULL;*/
static pcre        *ircRegex = NULL;
#if YAF_ENABLE_DPI
static pcre        *ircDPIRegex = NULL;
#endif


/**
 * static local functions
 *
 */
#if IRCDEBUG
static int
ycDebugBinPrintf(
    uint8_t   *data,
    uint16_t   size);
#endif /* if IRCDEBUG */

/**
 * ydpScanPayload
 *
 * scans a given payload to see if it conforms to our idea of what IRC traffic
 * looks like.
 *
 * @param payload pointer to the payload data
 * @param payloadSize the size of the payload parameter
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 * @return 0 for no match IRC_PORT_NUMBER (194) for a match
 *
 */
uint16_t
ydpScanPayload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    int rc;
#   define NUM_CAPT_VECTS 60
    int vects[NUM_CAPT_VECTS];

    rc = pcre_exec(ircMsgRegex, NULL, (char *)payload, payloadSize,
                   0, 0, vects, NUM_CAPT_VECTS);

    /*if (rc <= 0) {
     *  rc = pcre_exec(ircJoinRegex, NULL, (char *)payload, payloadSize,
     *                 0, 0, vects, NUM_CAPT_VECTS);
     *                 }*/
    if (rc <= 0) {
        rc = pcre_exec(ircRegex, NULL, (char *)payload, payloadSize,
                       0, 0, vects, NUM_CAPT_VECTS);
    }

    /** at some point in the future, this is the place to extract protocol
     *  information like message targets and join targets, etc.*/

#if YAF_ENABLE_DPI
    if (rc > 0) {
        ydRunPluginRegex(flow, payload, payloadSize, ircDPIRegex, 0,
                          202, IRC_PORT);
    }
#endif /* if YAF_ENABLE_DPI */

    if (rc > 0) {
        return IRC_PORT;
    }

    return 0;
}

#if YAF_ENABLE_DPI
void *
ydpProcessDPI(
    ypDPIFlowCtx_t       *flowContext,
    fbSubTemplateList_t  *stl,
    yfFlow_t             *flow,
    uint8_t               fwdcap,
    uint8_t               totalcap)
{
    return ydProcessGenericPlugin(flowContext, stl, flow, fwdcap, totalcap,
                                  YAF_IRC_FLOW_TID, ircTemplate,
                                  "ircTextMessage");
}

gboolean ydpAddTemplates(
    fbSession_t  *session)
{
    fbTemplateInfo_t *mdInfo;
    const fbInfoElement_t *bl_element;

    mdInfo = fbTemplateInfoAlloc();

    fbTemplateInfoInit(
        mdInfo, YAF_IRC_FLOW_NAME, YAF_IRC_FLOW_DESC, 194,
        FB_TMPL_MD_LEVEL_1);

    /* ruleset does not contain IE information, add metadata manually */
    /*fbTemplateInfoAddBasicList(mdInfo, IANA_ENT, IANA_BASICLIST_IE,
     * CERT_PEN, 125); */
    bl_element = ydLookupNamedBlByID(CERT_PEN, 125);
    if (bl_element) {
        fbTemplateInfoAddBasicList(mdInfo, bl_element->ent, bl_element->num,
                                   CERT_PEN, 125);
    }

    if (!ydInitTemplate(&ircTemplate, session, yaf_irc_spec,
                        mdInfo, YAF_IRC_FLOW_TID, 0))
    {
        return FALSE;
    }
    return TRUE;
}

void ydpFreeRec(
    ypDPIFlowCtx_t  *flowContext)
{
    yfIRCFlow_t *rec = (yfIRCFlow_t *)flowContext->rec;
    fbBasicListClear(&(rec->ircMsg));
}
#endif  /* YAF_ENABLE_DPI */


/**
 * ydpInitialize
 *
 * this initializes the PCRE expressions needed to search the payload for
 * IRC
 *
 *
 * @sideeffect sets the initialized flag on success
 *
 * @return 1 if initialization is complete correctly, 0 otherwise
 */
int
ydpInitialize(
    int             argc,
    char           *argv[],
    uint16_t        applabel,
    gboolean        applabelOnly,
    GError        **err)
{
    const char  ircMsgRegexString[] = "^(?:(:[^: \\n\\r]+)(?:\\ ))?"
        "(PRIVMSG|NOTICE) \\ "
        "([^: \\n\\r]+|:.*) (?:\\ )"
        "([^: \\n\\r]+\\ |:.*)";
    /*const char ircJoinRegexString[] = "^(?:(:[^\\: \\n\\r]+)(?:\\ ))?"
     * "((JOIN) \\ [^: \\n\\r]+\\ |:.*)\\s";*/
    const char ircRegexString[] = "^((?:(:[^: \\n\\r]+)(?:\\ ))?"
        "(PASS|OPER|QUIT|SQUIT|NICK"
        "|MODE|USER|SERVICE|JOIN|NAMES|INVITE"
        "|PART|TOPIC|LIST|KICK|PRIVMSG|NOTICE"
        "|MOTD|STATS|CONNECT|INFO|LUSERS|LINKS"
        "|TRACE|VERSION|TIME|ADMIN|SERVLIST"
        "|SQUERY|WHO|WHOWAS|WHOIS|KILL|PING"
        "|PONG|ERROR|AWAY|DIE|SUMMON|REHASH"
        "|RESTART|USERS|USERHOST)[ a-zA-Z0-9$#.:*\"]*)"
        "(?:[\\r\\n])";

#if YAF_ENABLE_DPI
    const char ircDPIRegexString[] = "((\\d{3}|PASS|OPER|QUIT|SQUIT|NICK"
        "|MODE|USER|SERVICE|JOIN|NAMES|INVITE"
        "|PART|TOPIC|LIST|KICK|PRIVMSG"
        "|MOTD|STATS|CONNECT|INFO|LUSERS|LINKS"
        "|TRACE|VERSION|TIME|ADMIN|SERVLIST"
        "|SQUERY|WHO|WHOWAS|WHOIS|KILL|PING"
        "|PONG|ERROR|AWAY|DIE|SUMMON|REHASH"
        "|RESTART|USERS|USERHOST|PROTOCTL) "
        "[-a-zA-Z0-9$#.:*\" ]*)(?:[\\r\\n])";
#endif

    ircRegex = ydPcreCompile(ircRegexString, PCRE_EXTENDED | PCRE_ANCHORED);
    ircMsgRegex = ydPcreCompile(ircMsgRegexString,
                                PCRE_EXTENDED | PCRE_ANCHORED);
    /*ircJoinRegex =
     * ydPcreCompile(ircJoinRegexString, PCRE_EXTENDED|PCRE_ANCHORED); */

    if (NULL == ircRegex || NULL == ircMsgRegex) {
        return 0;
    }

#if YAF_ENABLE_DPI
    ircDPIRegex = ydPcreCompile(ircDPIRegexString, PCRE_MULTILINE);
    if (NULL == ircDPIRegex) {
        return 0;
    }
#endif

    return 1;
}


#if IRCDEBUG
static int
ycDebugBinPrintf(
    uint8_t   *data,
    uint16_t   size)
{
    uint16_t loop;
    int      numPrinted = 0;

    for (loop = 0; loop < size; loop++) {
        if (isprint(*(data + loop)) && !iscntrl(*(data + loop))) {
            printf("%c", *(data + loop));
        } else {
            printf(".");
        }
        if ('\n' == *(data + loop) || '\r' == *(data + loop)
            || '\0' == *(data + loop))
        {
            break;
        }
        numPrinted++;
    }

    return numPrinted;
}
#endif /* if IRCDEBUG */
