/*
 *  Copyright 2007-2022 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/**
 *  @internal
 *
 *  @file mysqlplugin.c
 *
 *  @brief this is a protocol classifier for the MySQL protocol (MySQL)
 *
 *  MySQL
 *
 *  @ href="http://forge.mysql.com/wiki/MySQL_Internals_ClientServer_Protocol"
 *  ------------------------------------------------------------------------
 *  Authors: Emily Ecoff
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 3.0.0
 *
 *  Copyright 2022 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Released under a GNU GPL 2.0-style license, please see license.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
 *  Trademark Office by Carnegie Mellon University.
 *
 *  This Software includes and/or makes use of Third-Party Software subject
 *  to its own license.
 *
 *  DM22-0007
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */


#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <arpa/inet.h>

#include <yaf/yafDPIPlugin.h>
#if YAF_ENABLE_DPI

#define YAF_MYSQL_FLOW_TID      0xCE0C
#define YAF_MYSQL_FLOW_NAME     "yaf_mysql"
#define YAF_MYSQL_FLOW_DESC     NULL

#define YAF_MYSQLTXT_FLOW_TID   0xCE0D
#define YAF_MYSQLTXT_FLOW_NAME  "yaf_mysql_txt"
#define YAF_MYSQLTXT_FLOW_DESC  NULL

static fbInfoElementSpec_t yaf_mysql_spec[] = {
    {"mysqlCommandTextCodeList",   FB_IE_VARLEN, 0 },
    {"mysqlUsername",              FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

typedef struct yfMySQLFlow_st {
    fbSubTemplateList_t   mysqlList;
    fbVarfield_t          mysqlUsername;
} yfMySQLFlow_t;

static fbInfoElementSpec_t yaf_mysql_txt_spec[] = {
    {"mysqlCommandText",           FB_IE_VARLEN, 0 },
    {"mysqlCommandCode",           1, 0 },
    {"paddingOctets",              7, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

typedef struct yfMySQLTxtFlow_st {
    fbVarfield_t   mysqlCommandText;
    uint8_t        mysqlCommandCode;
    uint8_t        padding[7];
} yfMySQLTxtFlow_t;

static fbTemplate_t     *mysqlTemplate;
static fbTemplate_t     *mysqlTxtTemplate;
#endif

#define MYSQL_PORT_NUMBER 3306


/**
 * ydpScanPayload
 *
 * returns MYSQL_PORT_NUMBER if the passed in payload matches
 * a MySQL Server Greeting packet
 *
 * @param payload the packet payload
 * @param payloadSize size of the packet payload
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 *
 * return 0 if no match
 */
uint16_t
ydpScanPayload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    uint16_t payloadOffset = 0;
    uint32_t fillerOffset = 0;
    int      i = 0;
    uint8_t  packetNumber;
    uint32_t packetLength;
    uint8_t  temp;

    if (0 == payloadSize) {
        return 0;
    }

    packetLength = ((*(uint32_t *)payload)) & 0x00FFFFFF;

    payloadOffset += 3;
    if (packetLength < 49 || payloadOffset > payloadSize ||
        packetLength > payloadSize)
    {
        return 0;
    }

    packetNumber = *(payload + payloadOffset);

    payloadOffset++;

    if (packetNumber > 1) {
        return 0;
    }

    if (payloadOffset > payloadSize) {
        return 0;
    }

    if (packetNumber == 0) {
        /* Server Greeting */
        /*protoVersion = *(payload + payloadOffset);*/
        payloadOffset++;

        /* Version would be here - str until null*/

        /* Beginning of 0x00 fillers */
        fillerOffset = packetLength - 26 + 4;

        if (fillerOffset + 13 > payloadSize) {
            return 0;
        }

        for (i = 0; i < 13; i++) {
            temp = *(payload + fillerOffset + i);
            if (temp != 0) {
                return 0;
            }
        }
    } else {
        /* Client Authentication */
        /* Client Capabilities && Extended Capabilities*/
        payloadOffset += 4;

        /* Max Packet Size + 1 for Charset*/
        payloadOffset += 5;

        if ((size_t)payloadOffset + 23 > payloadSize) {
            return 0;
        }

        for (i = 0; i < 23; i++) {
            temp = *(payload + payloadOffset);
            if (temp != 0) {
                return 0;
            }
            payloadOffset++;
        }

#if YAF_ENABLE_DPI
        /* Here's the Username */
        i = 0;
        while ((payloadOffset < packetLength) &&
               ((size_t)payloadOffset + i < payloadSize))
        {
            if (*(payload + payloadOffset + i)) {
                i++;
            } else {
                break;
            }
        }

        ydRunPluginRegex(flow, payload, i, NULL, payloadOffset, 223,
                          MYSQL_PORT_NUMBER);

        /* Rest of pkt is password. Add 4 for pkt len & pkt num*/
        payloadOffset = packetLength + 4;

        if (packetLength > payloadSize) {
            return MYSQL_PORT_NUMBER;
        }

        /* Check for more packets */
        while (payloadOffset < payloadSize) {
            packetLength =
                (*(uint32_t *)(payload + payloadOffset)) & 0x00FFFFFF;

            if (packetLength > payloadSize) {
                return MYSQL_PORT_NUMBER;
            }

            payloadOffset += 4; /* add one for packet number */

            if (payloadOffset > payloadSize || packetLength == 0) {
                return MYSQL_PORT_NUMBER;
            }

            packetNumber = *(payload + payloadOffset);

            payloadOffset++;

            /* The text of the command follows */
            i = (packetLength - 1);

            if ((size_t)payloadOffset + i > payloadSize) {
                return MYSQL_PORT_NUMBER;
            }

            ydRunPluginRegex(flow, payload, i, NULL, payloadOffset,
                              packetNumber,
                              MYSQL_PORT_NUMBER);

            payloadOffset += i;
        }

#endif /* if YAF_ENABLE_DPI */
    }

    return MYSQL_PORT_NUMBER;
}


#if YAF_ENABLE_DPI
void *
ydpProcessDPI(
    ypDPIFlowCtx_t       *flowContext,
    fbSubTemplateList_t  *stl,
    yfFlow_t             *flow,
    uint8_t               fwdcap,
    uint8_t               totalcap)
{
    yfDPIData_t      *dpi = flowContext->dpi;
    yfMySQLFlow_t    *rec = NULL;
    yfMySQLTxtFlow_t *mysql = NULL;
    yfFlowVal_t      *val;
    uint8_t           count;
    uint8_t           start = flowContext->startOffset;
    int total = 0;

    g_assert(fwdcap <= totalcap);
    rec = (yfMySQLFlow_t *)fbSubTemplateListInit(stl, 3, YAF_MYSQL_FLOW_TID,
                                                 mysqlTemplate, 1);
    if (!flow->rval.payload) {
        totalcap = fwdcap;
    }

    for (count = start; count < totalcap; ++count) {
        /* since we test dpacketID < 29(0x1d), the != 223 is redundant.  did
         * not want to remove before confirming the test is correct. */
        if ((dpi[count].dpacketID != 223) && (dpi[count].dpacketID < 0x1d)) {
            total++;
        }
    }

    mysql = (yfMySQLTxtFlow_t *)fbSubTemplateListInit(
        &(rec->mysqlList), 3, YAF_MYSQLTXT_FLOW_TID, mysqlTxtTemplate, total);
    val = &flow->val;
    for (count = start; count < totalcap && mysql != NULL; ++count) {
        if (count == fwdcap) {
            val = &flow->rval;
        }
        /* MySQL Username */
        if (dpi[count].dpacketID == 223) {
            rec->mysqlUsername.buf = val->payload + dpi[count].dpacketCapt;
            rec->mysqlUsername.len = dpi[count].dpacketCaptLen;
        } else {
            mysql->mysqlCommandCode = dpi[count].dpacketID;
            mysql->mysqlCommandText.buf = val->payload + dpi[count].dpacketCapt;
            mysql->mysqlCommandText.len = dpi[count].dpacketCaptLen;
            mysql = fbSubTemplateListGetNextPtr(&(rec->mysqlList), mysql);
        }
    }

    return (void *)rec;
}

gboolean ydpAddTemplates(
    fbSession_t  *session)
{
    fbTemplateInfo_t *mdInfo;

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_MYSQL_FLOW_NAME, YAF_MYSQL_FLOW_DESC, 3306,
        FB_TMPL_MD_LEVEL_1);

    if (!ydInitTemplate(&mysqlTemplate, session, yaf_mysql_spec,
                        mdInfo, YAF_MYSQL_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_MYSQLTXT_FLOW_NAME, YAF_MYSQLTXT_FLOW_DESC, 3306,
        YAF_MYSQL_FLOW_TID);

    if (!ydInitTemplate(&mysqlTxtTemplate, session, yaf_mysql_txt_spec,
                        mdInfo, YAF_MYSQLTXT_FLOW_TID, 0))
    {
        return FALSE;
    }
    return TRUE;
}

void ydpFreeRec(
    ypDPIFlowCtx_t  *flowContext)
{
    yfMySQLFlow_t *rec = (yfMySQLFlow_t *)flowContext->rec;

    fbSubTemplateListClear(&(rec->mysqlList));
}
#endif  /* YAF_ENABLE_DPI */
