/*
 *  Copyright 2007-2022 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/**
 *  @internal
 *
 *  @INTERNAL
 *
 *  @file pop3plugin.c
 *
 *  this provides POP3 payload packet recognition for use within YAF
 *  It is based on RFC 1939 and some random limited packet capture.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Dan Ruef, Emily Ecoff
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 3.0.0
 *
 *  Copyright 2022 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Released under a GNU GPL 2.0-style license, please see license.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
 *  Trademark Office by Carnegie Mellon University.
 *
 *  This Software includes and/or makes use of Third-Party Software subject
 *  to its own license.
 *
 *  DM22-0007
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>
#include <pcre.h>

#include <yaf/yafDPIPlugin.h>
#if YAF_ENABLE_DPI

#define YAF_POP3_FLOW_TID   0xC300
#define YAF_POP3_FLOW_NAME  "yaf_pop3"
#define YAF_POP3_FLOW_DESC  NULL

static fbInfoElementSpec_t yaf_pop3_spec[] = {
    {"pop3TextMessageList",     FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

typedef struct yfPOP3Flow_st {
    fbBasicList_t   pop3msg;
} yfPOP3Flow_t;

static fbTemplate_t     *pop3Template;
#endif  /* YAF_ENABLE_DPI */

#define POP3DEBUG 0
#define POP3_PORT 110

/**
 * the compiled regular expressions, and related
 * flags
 *
 */
static pcre        *pop3RegexApplabel = NULL;
#if YAF_ENABLE_DPI
static pcre        *pop3RegexRequest  = NULL;
static pcre        *pop3RegexResponse = NULL;
#endif


/**
 * static local functions
 *
 */

#if POP3DEBUG
static int
ycDebugBinPrintf(
    uint8_t   *data,
    uint16_t   size);
#endif /* if POP3DEBUG */

/**
 * ydpScanPayload
 *
 * scans a given payload to see if it conforms to our idea of what POP3 traffic
 * looks like.
 *
 *
 *
 * @param payload pointer to the payload data
 * @param payloadSize the size of the payload parameter
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 * @return 0 for no match POP3_PORT_NUMBER (110) for a match
 *
 */
uint16_t
ydpScanPayload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    int      rc;
#   define NUM_CAPT_VECTS 60
    int      vects[NUM_CAPT_VECTS];

    rc = pcre_exec(pop3RegexApplabel, NULL, (char *)payload, payloadSize, 0,
                   0, vects, NUM_CAPT_VECTS);
    if (rc <= 0) {
        return 0;
    }

#if YAF_ENABLE_DPI
    if (rc == 2) {
        /* server side */
        ydRunPluginRegex(flow, payload, payloadSize, pop3RegexResponse, 0,
                         111, POP3_PORT);
    } else {
        /* client side */
        ydRunPluginRegex(flow, payload, payloadSize, pop3RegexRequest, 0,
                         110, POP3_PORT);
    }
#endif /* if YAF_ENABLE_DPI */

    return POP3_PORT;
}

#if YAF_ENABLE_DPI
void *
ydpProcessDPI(
    ypDPIFlowCtx_t       *flowContext,
    fbSubTemplateList_t  *stl,
    yfFlow_t             *flow,
    uint8_t               fwdcap,
    uint8_t               totalcap)
{
    return ydProcessGenericPlugin(flowContext, stl, flow, fwdcap, totalcap,
                                  YAF_POP3_FLOW_TID, pop3Template,
                                  "pop3TextMessage");
}

gboolean ydpAddTemplates(
    fbSession_t  *session)
{
    fbTemplateInfo_t *mdInfo;
    const fbInfoElement_t *bl_element;

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_POP3_FLOW_NAME, YAF_POP3_FLOW_DESC, 124,
        FB_TMPL_MD_LEVEL_1);

    /* ruleset does not contain IE information, add metadata manually */
    bl_element = ydLookupNamedBlByID(CERT_PEN, 124);
    if (bl_element) {
        fbTemplateInfoAddBasicList(mdInfo, bl_element->ent, bl_element->num,
                                   CERT_PEN, 124);
    }

    if (!ydInitTemplate(&pop3Template, session, yaf_pop3_spec,
                        mdInfo, YAF_POP3_FLOW_TID, 0))
    {
        return FALSE;
    }
    return TRUE;
}

void ydpFreeRec(
    ypDPIFlowCtx_t  *flowContext)
{
    yfPOP3Flow_t *rec = (yfPOP3Flow_t *)flowContext->rec;

    fbBasicListClear(&(rec->pop3msg));
}
#endif  /* YAF_ENABLE_DPI */

/**
 * ydpInitialize
 *
 * this initializes the PCRE expressions needed to search the payload for
 * POP3
 *
 *
 * @sideeffect sets the initialized flag on success
 *
 * @return 1 if initialization is complete correctly, 0 otherwise
 */
int
ydpInitialize(
    int             argc,
    char           *argv[],
    uint16_t        applabel,
    gboolean        applabelOnly,
    GError        **err)
{
#if YAF_ENABLE_DPI
    /* capture everything the client says */
    const char  pop3StringRequest[] =  "(?im)^[ \\t]*([!-~][ !-~]+)";

    /* capture the first line of each response */
    const char  pop3StringResponse[] = "(?m)^((?:\\+OK|-ERR)[ -~]*)";
#endif
    /* used to determine if this connection looks like POP3; capture the
     * response to distinguish the server from the client */
    const char  pop3StringApplabel[] =
        "(?i)^\\s*(?:(?:CAPA\\b|AUTH\\s(?:KERBEROS_V|GSSAPI|SKEY)|"
        "UIDL\\b|APOP\\s|USER\\s)|(\\+OK\\b|-ERR\\b))";

    pop3RegexApplabel = ydPcreCompile(pop3StringApplabel, 0);
    if (NULL == pop3RegexApplabel) {
        return 0;
    }

#if !YAF_ENABLE_DPI
    return 1;
#else
    pop3RegexRequest = ydPcreCompile(pop3StringRequest, 0);
    pop3RegexResponse = ydPcreCompile(pop3StringResponse, 0);

    return (pop3RegexRequest && pop3RegexResponse);
#endif  /* YAF_ENABLE_DPI */
}


#if POP3DEBUG
static int
ycDebugBinPrintf(
    uint8_t   *data,
    uint16_t   size)
{
    uint16_t loop;
    int      numPrinted = 0;

    for (loop = 0; loop < size; loop++) {
        if (isprint(*(data + loop)) && !iscntrl(*(data + loop))) {
            printf("%c", *(data + loop));
        } else {
            printf(".");
        }
        if ('\n' == *(data + loop) || '\r' == *(data + loop)
            || '\0' == *(data + loop))
        {
            break;
        }
        numPrinted++;
    }

    return numPrinted;
}
#endif /* if POP3DEBUG */
