/*
 *  Copyright 2007-2022 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/**
 *  @internal
 *
 *  @file tlsplugin.c
 *
 *
 *  This recognizes SSL & TLS packets
 *
 *  Remember to update proxyplugin.c with any changes.
 *  ------------------------------------------------------------------------
 *  Authors: Chris Inacio, Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  YAF 3.0.0
 *
 *  Copyright 2022 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Released under a GNU GPL 2.0-style license, please see license.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
 *  Trademark Office by Carnegie Mellon University.
 *
 *  This Software includes and/or makes use of Third-Party Software subject
 *  to its own license.
 *
 *  DM22-0007
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define _YAF_SOURCE_
#include <yaf/autoinc.h>
#include <yaf/yafcore.h>
#include <yaf/decode.h>

#include <yaf/yafDPIPlugin.h>


typedef struct yf_asn_tlv_st {
    uint8_t   class : 2;
    uint8_t   p_c   : 1;
    uint8_t   tag   : 5;
} yf_asn_tlv_t;


#if YAF_ENABLE_DPI
#define SSL_CERT_EXPORT_FLAG    0x02

/* YAF_SSL_FLOW_TID, yaf_ssl_spec, yfSSLFlow_t, sslTemplate, "yaf_ssl" */
#define YAF_SSL_FLOW_TID      0xCA0A
#define YAF_SSL_FLOW_NAME     "yaf_ssl"
#define YAF_SSL_FLOW_DESC     NULL

/* YAF_SSL_CERT_FLOW_TID, yaf_cert_spec, yfSSLCertFlow_t, sslCertTemplate,
 * "yaf_ssl_cert" */
#define YAF_SSL_CERT_FLOW_TID 0xCA0B
#define YAF_SSL_CERT_FLOW_NAME "yaf_ssl_cert"
#define YAF_SSL_CERT_FLOW_DESC NULL

/* YAF_SSL_SUBCERT_FLOW_TID, yaf_subssl_spec, yfSSLObjValue_t, sslSubTemplate,
 * "yaf_ssl_subcert" */
#define YAF_SSL_SUBCERT_FLOW_TID 0xCE14
#define YAF_SSL_SUBCERT_FLOW_NAME "yaf_ssl_subcert"
#define YAF_SSL_SUBCERT_FLOW_DESC NULL

/* YAF_SSL_FLOW_TID, yaf_ssl_spec, yfSSLFlow_t, sslTemplate, "yaf_ssl" */
static fbInfoElementSpec_t yaf_ssl_spec[] = {
    {"sslCipherList",             FB_IE_VARLEN, 0 }, /*list of ciphers 32bit */
    {"sslBinaryCertificateList",  FB_IE_VARLEN, SSL_CERT_EXPORT_FLAG },
    /* used by SM to label SSL DPI Level 1 */
    {"sslServerName",             FB_IE_VARLEN, 0 },
    {"sslCertList",               FB_IE_VARLEN, 0 }, /* list of certs */
    /* used by SM to label SSL DPI Level 1 */
    {"sslServerCipher",           4, 0 }, /*cipher suite in server hello */
    {"sslClientVersion",          1, 0 }, /* protocol version, 2 ssl, 3 tls */
    {"sslCompressionMethod",      1, 0 }, /*compression method in serv hello*/
    {"sslRecordVersion",          2, 0 }, /* message version */
    FB_IESPEC_NULL
};

/* YAF_SSL_FLOW_TID, yaf_ssl_spec, yfSSLFlow_t, sslTemplate, "yaf_ssl" */
typedef struct yfSSLFlow_st {
    fbBasicList_t         sslCipherList;
    fbBasicList_t         sslBinaryCertificateList;
    fbVarfield_t          sslServerName;
    fbSubTemplateList_t   sslCertList;
    uint32_t              sslServerCipher;
    uint8_t               sslClientVersion;
    uint8_t               sslCompressionMethod;
    uint16_t              sslVersion;
} yfSSLFlow_t;


/* YAF_SSL_CERT_FLOW_TID, yaf_cert_spec, yfSSLCertFlow_t, sslCertTemplate,
 * "yaf_ssl_cert" */
static fbInfoElementSpec_t yaf_cert_spec[] = {
    {"sslIssuerFieldList",          FB_IE_VARLEN, 0 },
    {"sslSubjectFieldList",         FB_IE_VARLEN, 0 },
    {"sslExtensionFieldList",       FB_IE_VARLEN, 0 },
    /* used by SM to label SSL DPI Level 2 */
    {"sslCertSignature",            FB_IE_VARLEN, 0 },
    /* used by SM to label SSL DPI Level 2 */
    {"sslCertSerialNumber",         FB_IE_VARLEN, 0 },
    {"sslCertValidityNotBefore",    FB_IE_VARLEN, 0 },
    {"sslCertValidityNotAfter",     FB_IE_VARLEN, 0 },
    {"sslPublicKeyAlgorithm",       FB_IE_VARLEN, 0 },
    {"sslPublicKeyLength",          2, 0 },
    /* used by SM to label SSL DPI Level 2 */
    {"sslCertVersion",              1, 0 },
    {"paddingOctets",               5, 0 }, /* TODO: re-enable when downstream is ready */
    {"sslCertificateHash",          FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* YAF_SSL_CERT_FLOW_TID, yaf_cert_spec, yfSSLCertFlow_t, sslCertTemplate,
 * "yaf_ssl_cert" */
typedef struct yfSSLCertFlow_st {
    fbSubTemplateList_t   issuer;
    fbSubTemplateList_t   subject;
    fbSubTemplateList_t   extension;
    fbVarfield_t          sig;
    fbVarfield_t          serial;
    fbVarfield_t          not_before;
    fbVarfield_t          not_after;
    fbVarfield_t          pkalg;
    uint16_t              pklen;
    uint8_t               version;
    uint8_t               padding[5];
    fbVarfield_t          hash;
} yfSSLCertFlow_t;

/* YAF_SSL_SUBCERT_FLOW_TID, yaf_subssl_spec, yfSSLObjValue_t, sslSubTemplate,
 * "yaf_ssl_subcert" */
static fbInfoElementSpec_t yaf_subssl_spec[] = {
    {"sslObjectValue",              FB_IE_VARLEN, 0 },
    {"sslObjectType",               1, 0 },
    {"paddingOctets",               7, YAF_INT_PADDING_FLAG },
    FB_IESPEC_NULL
};

/* YAF_SSL_SUBCERT_FLOW_TID, yaf_subssl_spec, yfSSLObjValue_t, sslSubTemplate,
 * "yaf_ssl_subcert" */
typedef struct yfSSLObjValue_st {
    fbVarfield_t   obj_value;
    uint8_t        obj_id;
    uint8_t        padding[7];
} yfSSLObjValue_t;

/* YAF_SSL_FLOW_TID, yaf_ssl_spec, yfSSLFlow_t, sslTemplate, "yaf_ssl" */
static fbTemplate_t     *sslTemplate;

/* YAF_SSL_CERT_FLOW_TID, yaf_cert_spec, yfSSLCertFlow_t, sslCertTemplate,
 * "yaf_ssl_cert" */
static fbTemplate_t     *sslCertTemplate;

/* YAF_SSL_SUBCERT_FLOW_TID, yaf_subssl_spec, yfSSLObjValue_t, sslSubTemplate,
 * "yaf_ssl_subcert" */
static fbTemplate_t     *sslSubTemplate;

/* When TRUE, causes the complete binary X.509 certificate to be exported.
 * Set by the arguments to ydpInitialize(). */
static gboolean  full_cert_export = FALSE;

/* When TRUE, causes the hash of the X.509 certificate to be exported.  Set by
 * the arguments to ydpInitialize(). */
static gboolean  cert_hash_export = FALSE;

/* When TRUE, turns off the default DPI processing of the certificates.  Set
 * by the arguments to ydpInitialize(). */
static gboolean  ssl_dpi_off = FALSE;
#endif  /* YAF_ENABLE_DPI */

/* ASN.1 Tag Numbers (for SSL) */
#define CERT_BOOL               0x01
#define CERT_INT                0x02
#define CERT_BITSTR             0x03
#define CERT_OCTSTR             0x04
#define CERT_NULL               0x05
/* Object Identifer */
#define CERT_OID                0x06
/* Start of Sequence */
#define CERT_SEQ                0x10
/* Start of Set */
#define CERT_SET                0x11
/* Printable String */
#define CERT_PRINT              0x13
/* UTC Time */
#define CERT_TIME               0x17
#define CERT_EXPLICIT           0xa0
/* ASN.1 P/C Bit (primitive, constucted) */
#define CERT_PRIM               0x00
#define CERT_CONST              0x01
/* ASN.1 Length 0x81 is length follows in 1 byte */
#define CERT_1BYTE              0x81
/* ASN.1 Length 0x82 is length follows in 2 bytes */
#define CERT_2BYTE              0x82
#define CERT_IDCE               0x551D
#define CERT_IDAT               0x5504
/* {iso(1) member-body (2) us(840) rsadsi(113459) pkcs(1) 9} */
#define CERT_PKCS               0x2A864886
/* 0.9.2342.19200300.100.1.25 */
#define CERT_DC                 0x09922689


/* this might be more - but I have to have a limit somewhere */
#define MAX_CERTS 10

/** defining the header structure for SSLv2 is pointless, because the
 *  first field of the record is variable length, either 2 or 3 bytes
 *  meaning that the first step has to be to figure out how far offset
 *  all of the other fields are.  Further, the client can send a v2
 *  client_hello stating that it is v3/TLS 1.0 capable, and the server
 *  can respond with v3/TLS 1.0 record formats
 */


/** this defines the record header for SSL V3 negotiations,
 *  it also works for TLS 1.0 */
typedef struct sslv3RecordHeader_st {
    uint8_t    contentType;
    uint8_t    protocolMajor;
    uint8_t    protocolMinor;
    uint16_t   length;
} sslv3RecordHeader_t;

static gboolean
decodeSSLv2(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    uint16_t        offsetptr,
    uint8_t         datalength);

static gboolean
decodeTLSv1(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    uint16_t        offsetptr,
    uint8_t         datalength,
    uint8_t         type);

#define TLS_PORT_NUMBER  443

#define TLS_VERSION_1 0x0301
#define SSL_VERSION_2 0x0002
#define SSL_VERSION_3 0x0003
#define TLS_VERSION_11 0x0302
#define TLS_VERSION_12 0x0303
#define SSL_VERSION 0x0200

/**
 * SSL CERT Parsing
 *
 */

#if YAF_ENABLE_DPI
static gboolean
ypDecodeSSLCertificate(
    yfDPIContext_t   *ctx,
    yfSSLCertFlow_t **sslCert,
    const uint8_t    *payload,
    unsigned int      payloadSize,
    yfFlow_t         *flow,
    uint16_t          offsetptr);
#endif  /* YAF_ENABLE_DPI */

/**
 * ydpScanPayload
 *
 * the scanner for recognizing SSL/TLS packets
 *
 * @param payload the packet payload
 * @param payloadSize size of the packet payload
 * @param flow a pointer to the flow state structure
 * @param val a pointer to biflow state (used for forward vs reverse)
 *
 *
 * @return TLS_PORT_NUMBER
 *         otherwise 0
 */
uint16_t
ydpScanPayload(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    yfFlowVal_t    *val)
{
    uint8_t  ssl_length;
    uint8_t  ssl_msgtype;
    uint16_t tls_version;
    uint16_t offsetptr = 0;

    /* every SSL/TLS header has to be at least 2 bytes long... */
    /* we need 5 to determine message type and version */
    if (payloadSize < 5) {
        return 0;
    }

    /*understanding how to determine between SSLv2 and SSLv3/TLS is "borrowed"
     * from OpenSSL payload byte 0 for v2 is the start of the length field, but
     * its MSb is always reserved to tell us how long the length field is, and
     * in some cases, the second MSb is reserved as well */

    /* when length is 2 bytes in size (MSb == 1), and the message type code is
     * 0x01 (client_hello) we know we're doing SSL v2 */
    if ((payload[0] & 0x80) && (0x01 == payload[2])) {
        ssl_length = ((payload[0] & 0x7F) << 8) | payload[1];

        if (ssl_length < 2) {
            return 0;
        }

        ssl_msgtype = 1;
        offsetptr += 3;

        /* this is the version from the handshake message */
        tls_version = ntohs(*(uint16_t *)(payload + offsetptr));
        offsetptr += 2;
        if (tls_version == TLS_VERSION_1 || tls_version == SSL_VERSION_2 ||
            tls_version == SSL_VERSION_3)
        {
            if (!decodeSSLv2(payload, payloadSize, flow, offsetptr,
                             ssl_length))
            {
                return 0;
            }
        } else {
            return 0;
        }

        /* SSLv2 (client_hello) */
#if YAF_ENABLE_DPI
        ydRunPluginRegex(flow, payload, 1, NULL, 2, 88, TLS_PORT_NUMBER);
        ydRunPluginRegex(flow, payload, 2, NULL, tls_version, 94,
                          TLS_PORT_NUMBER);
#endif
        return TLS_PORT_NUMBER;
    } else {
        if ((0x00 == (payload[0] & 0x80)) && (0x00 == (payload[0] & 0x40))
            && (0x01 == payload[3]))
        {
            /* this is ssl v2 but with a 3-byte header */
            /* the second MSb means the record is a data record */
            /* the fourth byte should be 1 for client hello */
            if ((payload[0] == 0x16) && (payload[1] == 0x03)) {
                /* this is most likely tls, not sslv2 */
                goto tls;
            }

            ssl_length = ((payload[0] * 0x3F) << 8) | payload[1];

            if (ssl_length < 3) {
                return 0;
            }
            offsetptr += 4;

            if ( ((size_t)offsetptr + 2) < payloadSize) {
                tls_version = ntohs(*(uint16_t *)(payload + offsetptr));
                offsetptr += 2;

                if (tls_version == TLS_VERSION_1 ||
                    tls_version == SSL_VERSION_2 ||
                    tls_version == SSL_VERSION_3)
                {
                    if (!decodeSSLv2(payload, payloadSize, flow, offsetptr,
                                     ssl_length))
                    {
                        return 0;
                    }
                } else {
                    return 0;
                }
            } else {
                return 0;
            }
#if YAF_ENABLE_DPI
            ydRunPluginRegex(flow, payload, 1, NULL, 2, 88,
                              TLS_PORT_NUMBER);
            ydRunPluginRegex(flow, payload, 2, NULL, tls_version, 94,
                              TLS_PORT_NUMBER);
#endif /* if YAF_ENABLE_DPI */
            return TLS_PORT_NUMBER;
        }
      tls:
        if (payloadSize >= 10) {
            /* payload[0] is handshake request [0x16]
             * payload[1] is ssl major version, sslv3 & tls is 3
             * payload[5] is handshake command, 1=client_hello,2=server_hello
             * payload[3,4] is length
             * payload[9] is the version from the record */
            if ((payload[0] == 0x16) && (payload[1] == 0x03) &&
                ((payload[5] == 0x01) || (payload[5] == 0x02)) &&
                (((payload[3] == 0) && (payload[4] < 5)) ||
                 (payload[9] == payload[1])))
            {
                ssl_msgtype = payload[5];
                ssl_length = payload[4];
                tls_version = ntohs(*(uint16_t *)(payload + 1));
                /* 1 for content type, 2 for version, 2 for length,
                 * 1 for handshake type*/
                offsetptr += 6;
                /* now we should be at record length */
                if (!decodeTLSv1(payload, payloadSize, flow, offsetptr,
                                 ssl_length, ssl_msgtype))
                {
                    return 0;
                }

                /* SSLv3 / TLS */
#if YAF_ENABLE_DPI
                ydRunPluginRegex(flow, payload, 1, NULL, 3, 88,
                                  TLS_PORT_NUMBER);
                ydRunPluginRegex(flow, payload, 2, NULL, tls_version, 94,
                                  TLS_PORT_NUMBER);
#endif /* if YAF_ENABLE_DPI */
                return TLS_PORT_NUMBER;
            }
        }
    }

    return 0;
}

#if YAF_ENABLE_DPI
void *
ydpProcessDPI(
    ypDPIFlowCtx_t       *flowContext,
    fbSubTemplateList_t  *stl,
    yfFlow_t             *flow,
    uint8_t               fwdcap,
    uint8_t               totalcap)
{
    yfDPIData_t     *dpi = flowContext->dpi;
    yfDPIContext_t  *ctx = flowContext->yfctx;
    yfSSLFlow_t     *rec = NULL;
    yfSSLCertFlow_t *sslcert = NULL;
    fbInfoModel_t   *model = ydGetDPIInfoModel();
    int              count = flowContext->startOffset;
    int              total_certs = 0;
    uint32_t        *sslCiphers;
    const uint8_t   *payload = NULL;
    size_t           paySize = 0;
    uint8_t          totalIndex[YAF_MAX_CAPTURE_FIELDS];
    gboolean         ciphertrue = FALSE;
    int              i;
    fbVarfield_t    *sslfull = NULL;
    const fbInfoElement_t *sslCipherIE;
    const fbInfoElement_t *sslCertificateIE;

    rec = (yfSSLFlow_t *)fbSubTemplateListInit(stl, 3, YAF_SSL_FLOW_TID,
                                               sslTemplate, 1);
    sslCipherIE = fbInfoModelGetElementByName(model, "sslCipher");
    sslCertificateIE = fbInfoModelGetElementByName(model,
                                                   "sslBinaryCertificate");

    if (!flow->rval.payload) {
        totalcap = fwdcap;
    }

    while (count < totalcap) {
        if (count < fwdcap) {
            payload = flow->val.payload;
            paySize = flow->val.paylen;
        } else if (flow->rval.payload) {
            payload = flow->rval.payload;
            paySize = flow->rval.paylen;
        } else {
            count++;
            continue;
        }

        if (dpi[count].dpacketID == 91) {
            sslCiphers = (uint32_t *)fbBasicListInit(
                &rec->sslCipherList, 3, sslCipherIE,
                dpi[count].dpacketCaptLen / 2);
            for (i = 0; i < (dpi[count].dpacketCaptLen / 2); i++) {
                *sslCiphers = (uint32_t)ntohs(
                    *(uint16_t *)(payload + dpi[count].dpacketCapt + (i * 2)));
                if (!(sslCiphers = fbBasicListGetNextPtr(&rec->sslCipherList,
                                                         sslCiphers)))
                {
                    break;
                }
            }
            ciphertrue = TRUE;
        } else if (dpi[count].dpacketID == 90) {
            rec->sslCompressionMethod = *(payload + dpi[count].dpacketCapt);
        } else if (dpi[count].dpacketID == 88) {
            /* major version */
            if (!rec->sslClientVersion) {
                rec->sslClientVersion = dpi[count].dpacketCapt;
            }
        } else if (dpi[count].dpacketID == 94) {
            /* record version */
            rec->sslVersion = dpi[count].dpacketCapt;
        } else if (dpi[count].dpacketID == 89) {
            rec->sslServerCipher = ntohs(*(uint16_t *)(payload +
                                                       dpi[count].dpacketCapt));
        } else if (dpi[count].dpacketID == 92) {
            sslCiphers = (uint32_t *)fbBasicListInit(
                &rec->sslCipherList, 3, sslCipherIE,
                dpi[count].dpacketCaptLen / 3);
            for (i = 0; i < (dpi[count].dpacketCaptLen / 3); i++) {
                *sslCiphers = (ntohl(*(uint32_t *)(payload +
                                                   dpi[count].dpacketCapt +
                                                   (i * 3))) & 0xFFFFFF00) >> 8;
                if (!(sslCiphers = fbBasicListGetNextPtr(&rec->sslCipherList,
                                                         sslCiphers)))
                {
                    break;
                }
            }
            ciphertrue = TRUE;
        } else if (dpi[count].dpacketID == 93) {
            totalIndex[total_certs] = count;
            total_certs++;
        } else if (dpi[count].dpacketID == 95) {
            /* server Name */
            rec->sslServerName.buf =
                (uint8_t *)payload + dpi[count].dpacketCapt;
            rec->sslServerName.len = dpi[count].dpacketCaptLen;
        }

        count++;
    }

    if (!ciphertrue) {
        fbBasicListInit(&rec->sslCipherList, 3, sslCipherIE, 0);
    }

    if (ssl_dpi_off) {
        /* empty since we're doing full cert export */
        sslcert = (yfSSLCertFlow_t *)fbSubTemplateListInit(
            &rec->sslCertList, 3, YAF_SSL_CERT_FLOW_TID, sslCertTemplate, 0);
    } else {
        sslcert = ((yfSSLCertFlow_t *)fbSubTemplateListInit(
                       &rec->sslCertList, 3,
                       YAF_SSL_CERT_FLOW_TID, sslCertTemplate, total_certs));
        for (i = 0; i < total_certs; i++) {
            if (totalIndex[i] < fwdcap) {
                payload = flow->val.payload;
                paySize = flow->val.paylen;
            } else if (flow->rval.payload) {
                payload = flow->rval.payload;
                paySize = flow->rval.paylen;
            }
            if (!ypDecodeSSLCertificate(ctx, &sslcert, payload, paySize, flow,
                                        dpi[totalIndex[i]].dpacketCapt))
            {
                if (sslcert->issuer.tmpl == NULL) {
                    fbSubTemplateListInit(
                        &sslcert->issuer, 3,
                        YAF_SSL_SUBCERT_FLOW_TID, sslSubTemplate, 0);
                }
                if (sslcert->subject.tmpl == NULL) {
                    fbSubTemplateListInit(
                        &sslcert->subject, 3,
                        YAF_SSL_SUBCERT_FLOW_TID, sslSubTemplate, 0);
                }
                if (sslcert->extension.tmpl == NULL) {
                    fbSubTemplateListInit(
                        &sslcert->extension, 3,
                        YAF_SSL_SUBCERT_FLOW_TID, sslSubTemplate, 0);
                }
            }

            if (!(sslcert =
                      fbSubTemplateListGetNextPtr(&rec->sslCertList,
                                                  sslcert)))
            {
                break;
            }
        }
    }

    if (full_cert_export) {
        uint32_t sub_cert_len;
        uint32_t tot_bl_len = 0;
        uint32_t offset;
        sslfull = (fbVarfield_t *)fbBasicListInit(
            &rec->sslBinaryCertificateList, 3, sslCertificateIE, total_certs);
        for (i = 0; i < total_certs; i++) {
            if (totalIndex[i] < fwdcap) {
                payload = flow->val.payload;
                paySize = flow->val.paylen;
            } else if (flow->rval.payload) {
                payload = flow->rval.payload;
                paySize = flow->rval.paylen;
            }
            offset = dpi[totalIndex[i]].dpacketCapt;
            if (offset + 4 > paySize) {
                sslfull->len = 0;
                sslfull->buf = NULL;
                sslfull = (fbVarfield_t *)fbBasicListGetNextPtr(
                    &rec->sslBinaryCertificateList, sslfull);
                continue;
            }
            sub_cert_len =
                (ntohl(*(uint32_t *)(payload + offset)) & 0xFFFFFF00) >> 8;

            /* only continue if we have enough payload for the whole cert */
            if (offset + sub_cert_len > paySize) {
                sslfull->len = 0;
                sslfull->buf = NULL;
                sslfull = (fbVarfield_t *)fbBasicListGetNextPtr(
                    &rec->sslBinaryCertificateList, sslfull);
                continue;
            }

            sslfull->buf = (uint8_t *)payload + offset + 3;
            sslfull->len = sub_cert_len;
            tot_bl_len += sub_cert_len;
            sslfull = (fbVarfield_t *)fbBasicListGetNextPtr(
                &rec->sslBinaryCertificateList, sslfull);
        }

        if (!tot_bl_len) {
            fbBasicListClear(&rec->sslBinaryCertificateList);
            sslfull = (fbVarfield_t *)fbBasicListInit(
                &rec->sslBinaryCertificateList, 3, sslCertificateIE, 0);
        }
    }

    return (void *)rec;
}

gboolean ydpAddTemplates(
    fbSession_t  *session)
{
    fbTemplateInfo_t *mdInfo;
    const fbInfoElement_t *bl_element;

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_SSL_FLOW_NAME, YAF_SSL_FLOW_DESC, 443,
        FB_TMPL_MD_LEVEL_1);
    /* sslCipher */
    bl_element = ydLookupNamedBlByID(CERT_PEN, 185);
    if (bl_element) {
        fbTemplateInfoAddBasicList(mdInfo, bl_element->ent, bl_element->num,
                                   CERT_PEN, 185);
    }

    if (full_cert_export) {
        /* sslBinaryCertificate */
        bl_element = ydLookupNamedBlByID(CERT_PEN, 296);
        if (bl_element) {
            fbTemplateInfoAddBasicList(mdInfo, bl_element->ent,
                                       bl_element->num, CERT_PEN, 296);
        }

        if (!ydInitTemplate(&sslTemplate, session, yaf_ssl_spec,
                            mdInfo, YAF_SSL_FLOW_TID, SSL_CERT_EXPORT_FLAG))
        {
            return FALSE;
        }
    } else {
        if (!ydInitTemplate(&sslTemplate, session, yaf_ssl_spec,
                            mdInfo, YAF_SSL_FLOW_TID, 0))
        {
            return FALSE;
        }
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_SSL_CERT_FLOW_NAME, YAF_SSL_CERT_FLOW_DESC, 443,
        YAF_SSL_FLOW_TID);

    if (!ydInitTemplate(&sslCertTemplate, session, yaf_cert_spec,
                        mdInfo, YAF_SSL_CERT_FLOW_TID, 0))
    {
        return FALSE;
    }

    mdInfo = fbTemplateInfoAlloc();
    fbTemplateInfoInit(
        mdInfo, YAF_SSL_SUBCERT_FLOW_NAME, YAF_SSL_SUBCERT_FLOW_DESC, 443,
        YAF_SSL_CERT_FLOW_TID);

    if (!ydInitTemplate(&sslSubTemplate, session, yaf_subssl_spec,
                        mdInfo, YAF_SSL_SUBCERT_FLOW_TID, 0))
    {
        return FALSE;
    }

    return TRUE;
}

void
ydpFreeRec(
    ypDPIFlowCtx_t  *flowContext)
{
    yfSSLFlow_t     *rec = (yfSSLFlow_t *)flowContext->rec;
    yfSSLCertFlow_t *cert = NULL;

    while ((cert = fbSubTemplateListGetNextPtr(&rec->sslCertList, cert))) {
        fbSubTemplateListClear(&cert->issuer);
        fbSubTemplateListClear(&cert->subject);
        fbSubTemplateListClear(&cert->extension);
    }

    fbSubTemplateListClear(&rec->sslCertList);

    fbBasicListClear(&rec->sslCipherList);

    if (full_cert_export) {
        fbBasicListClear(&rec->sslBinaryCertificateList);
    }
}
#endif  /* YAF_ENABLE_DPI */

static gboolean
decodeTLSv1(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    uint16_t        offsetptr,
    uint8_t         datalength,
    uint8_t         type)
{
    uint32_t record_len;
    uint16_t header_len = offsetptr - 1;
    uint32_t cert_len, sub_cert_len;
    uint16_t cert_version;
    int      cert_count = 0;
    uint16_t ciphers = 0;
    uint16_t cipher_suite_len;
    uint8_t  session_len;
    uint8_t  compression_len;
    uint8_t  next_msg;
    uint16_t ext_len = 0;
    uint16_t ext_ptr;

    /* Both Client and Server Hello's start off the same way */
    /* 3 for Length, 2 for Version, 32 for Random, 1 for session ID Len*/
    if ((size_t)offsetptr + 39 > payloadSize) {
        return FALSE;
    }

    record_len = (ntohl(*(uint32_t *)(payload + offsetptr)) & 0xFFFFFF00) >> 8;

    offsetptr += 2;

    cert_version = ntohs(*(uint16_t *)(payload + offsetptr));

    offsetptr += 35; /* skip version & random */

    session_len = *(payload + offsetptr);

    offsetptr += session_len + 1;

    if ((size_t)offsetptr + 2 > payloadSize) {
        return FALSE;
    }

    if (type == 1) {
        /* Client Hello */

        cipher_suite_len = ntohs(*(uint16_t *)(payload + offsetptr));

        /* figure out number of ciphers by dividing by 2 */

        offsetptr += 2;

        if (cipher_suite_len > payloadSize) {
            return FALSE;
        }

        if ((size_t)offsetptr + cipher_suite_len > payloadSize) {
            return FALSE;
        }

        ciphers = offsetptr;

        /* cipher length */
        /* ciphers are here */
        offsetptr += cipher_suite_len;

        if ((size_t)offsetptr + 1 > payloadSize) {
            return FALSE;
        }

        compression_len = *(payload + offsetptr);

        offsetptr += compression_len + 1;

#if YAF_ENABLE_DPI
        ydRunPluginRegex(flow, payload, 2, NULL, cert_version, 94,
                          TLS_PORT_NUMBER);
        ydRunPluginRegex(flow, payload, cipher_suite_len, NULL, ciphers,
                          91, TLS_PORT_NUMBER);
#endif /* if YAF_ENABLE_DPI */
    } else if (type == 2) {
        /* Server Hello */
        if ((size_t)offsetptr + 3 > payloadSize) {
            return FALSE;
        }
        /* cipher is here */
#if YAF_ENABLE_DPI
        ydRunPluginRegex(flow, payload, 2, NULL, offsetptr, 89,
                          TLS_PORT_NUMBER);
#endif
        offsetptr += 2;
        /* compression method */
#if YAF_ENABLE_DPI
        ydRunPluginRegex(flow, payload, 2, NULL, cert_version, 94,
                          TLS_PORT_NUMBER);
        ydRunPluginRegex(flow, payload, 1, NULL, offsetptr, 90,
                          TLS_PORT_NUMBER);
#endif /* if YAF_ENABLE_DPI */
        offsetptr++;
    }

    if (((size_t)offsetptr - header_len) < record_len) {
        /* extensions? */
        ext_len = ntohs(*(uint16_t *)(payload + offsetptr));
        ext_ptr = offsetptr + 2;
        offsetptr += 2 + ext_len;
#if YAF_ENABLE_DPI
        /* only want Client Hello's server name */
        if (type == 1) {
            uint16_t sub_ext_len;
            uint16_t sub_ext_type;
            int      tot_ext = 0;

            while (ext_ptr < payloadSize && (tot_ext < ext_len)) {
                sub_ext_type = ntohs(*(uint16_t *)(payload + ext_ptr));
                ext_ptr += 2;
                sub_ext_len = ntohs(*(uint16_t *)(payload + ext_ptr));
                ext_ptr += 2;
                tot_ext += sizeof(uint16_t) + sizeof(uint16_t) + sub_ext_len;
                if (sub_ext_type != 0) {
                    ext_ptr += sub_ext_len;
                    continue;
                }
                if (sub_ext_len == 0) {
                    /* no server name listed */
                    break;
                }
                /* grab server name */
                /* jump past list length and type to get name length and name
                 * */
                ext_ptr += 3; /* 2 for length, 1 for type */
                sub_ext_len = ntohs(*(uint16_t *)(payload + ext_ptr));
                ext_ptr += 2;
                if ((ext_ptr + sub_ext_len) < payloadSize) {
                    ydRunPluginRegex(flow, payload, sub_ext_len, NULL, ext_ptr,
                                      95, TLS_PORT_NUMBER);
                }
                break;
            }
        }
#endif /* if YAF_ENABLE_DPI */
    }
    while (payloadSize > offsetptr) {
        next_msg = *(payload + offsetptr);
        if (next_msg == 11) {
            /* certificate */
            if ((size_t)offsetptr + 7 > payloadSize) {
                return TRUE; /* prob should be false */
            }

            offsetptr++;

            record_len = (ntohl(*(uint32_t *)(payload + offsetptr)) &
                          0xFFFFFF00) >> 8;
            offsetptr += 3;

            /* Total Cert Length */
            cert_len = (ntohl(*(uint32_t *)(payload + offsetptr)) &
                        0xFFFFFF00) >> 8;
            offsetptr += 3;

            while (payloadSize > ((size_t)offsetptr + 4)) {
                sub_cert_len = (ntohl(*(uint32_t *)(payload + offsetptr)) &
                                0xFFFFFF00) >> 8;

                if ((sub_cert_len > cert_len) || (sub_cert_len < 2)) {
                    /* it's at least got to have a version number */
                    return TRUE; /* prob should be false */
                } else if (sub_cert_len > payloadSize) {
                    /* just not enough room */
                    return TRUE;
                }

                /* offset of certificate */
                if (cert_count < MAX_CERTS) {
#if YAF_ENABLE_DPI
                    if (((size_t)offsetptr + sub_cert_len + 3) < payloadSize) {
                        ydRunPluginRegex(flow, payload, 1, NULL, offsetptr,
                                          93, TLS_PORT_NUMBER);
                    }
#endif /* if YAF_ENABLE_DPI */
                } else {
                    return TRUE;
                }

                cert_count++;
                offsetptr += 3 + sub_cert_len;
            }
        } else if (next_msg == 22) {
            /* 1 for type, 2 for version, 2 for length - we know it's long */
            offsetptr += 5;
        } else if (next_msg == 20 || next_msg == 21 || next_msg == 23) {
            offsetptr += 3; /* 1 for type, 2 for version */

            if (((size_t)offsetptr + 2) > payloadSize) {
                return TRUE; /* prob should be false */
            }

            record_len = ntohs(*(uint16_t *)(payload + offsetptr));

            if (record_len > payloadSize) {
                return TRUE;
            }

            offsetptr += record_len + 2;
        } else {
            return TRUE;
        }
    }

    return TRUE;
}


static gboolean
decodeSSLv2(
    const uint8_t  *payload,
    unsigned int    payloadSize,
    yfFlow_t       *flow,
    uint16_t        offsetptr,
    uint8_t         datalength)
{
    uint32_t record_len;
    uint16_t cipher_spec_length;
    uint16_t challenge_length;
    uint32_t cert_len, sub_cert_len;
    int      cert_count = 0;
    uint8_t  next_msg;

    if ((size_t)offsetptr + 6 > payloadSize) {
        return FALSE;
    }

    cipher_spec_length = ntohs(*(uint16_t *)(payload + offsetptr));

    /* cipher_spec_length */
    /* session length */

    offsetptr += 4;

    /* challenge length */
    challenge_length = ntohs(*(uint16_t *)(payload + offsetptr));

    offsetptr += 2;

    if ((size_t)offsetptr + cipher_spec_length > payloadSize) {
        return FALSE;
    }

    if (cipher_spec_length > payloadSize) {
        return FALSE;
    }

#if YAF_ENABLE_DPI
    ydRunPluginRegex(flow, payload, cipher_spec_length, NULL, offsetptr, 92,
                      TLS_PORT_NUMBER);
#endif
    offsetptr += cipher_spec_length + challenge_length;

    while (payloadSize > offsetptr) {
        next_msg = *(payload + offsetptr);

        if (next_msg == 11) {
            /* certificate */
            if ((size_t)offsetptr + 7 > payloadSize) {
                return TRUE; /* prob should be false */
            }

            offsetptr++;

            record_len = (ntohl(*(uint32_t *)(payload + offsetptr)) &
                          0xFFFFFF00) >> 8;
            offsetptr += 3;

            /* Total Cert Length */
            cert_len = (ntohl(*(uint32_t *)(payload + offsetptr)) &
                        0xFFFFFF00) >> 8;
            offsetptr += 3;

            while (payloadSize > ((size_t)offsetptr + 4)) {
                sub_cert_len = (ntohl(*(uint32_t *)(payload + offsetptr)) &
                                0xFFFFFF00) >> 8;

                if ((sub_cert_len > cert_len) || (sub_cert_len < 2)) {
                    /* it's at least got to have a version number */
                    return TRUE; /* prob should be false */
                } else if (sub_cert_len > payloadSize) {
                    /* just not enough room */
                    return TRUE;
                }

                /* offset of certificate */
                if (cert_count < MAX_CERTS) {
#if YAF_ENABLE_DPI
                    if (((size_t)offsetptr + sub_cert_len + 3) < payloadSize) {
                        ydRunPluginRegex(flow, payload, 1, NULL, offsetptr,
                                          93, TLS_PORT_NUMBER);
                    }
#endif /* if YAF_ENABLE_DPI */
                } else {
                    return TRUE;
                }

                cert_count++;
                offsetptr += 3 + sub_cert_len;
            }
        } else if (next_msg == 22) {
            /* 1 for type, 2 for version, 2 for length - we know it's long */
            offsetptr += 5;
        } else if (next_msg == 20 || next_msg == 21 || next_msg == 23) {
            offsetptr += 3; /* 1 for type, 2 for version */

            if (((size_t)offsetptr + 2) > payloadSize) {
                return TRUE; /* prob should be false */
            }

            record_len = ntohs(*(uint16_t *)(payload + offsetptr));

            if (record_len > payloadSize) {
                return TRUE;
            }

            offsetptr += record_len + 2;
        } else {
            return TRUE;
        }
    }

    return TRUE;
}


#if YAF_ENABLE_DPI
/*
 *  Decodes the length in `payload` at the current `offset`.  Moves `offset`
 *  to the final byte of the length and returns the length.
 *
 *  When the high bit of the byte at `offset` is not set, that single byte is
 *  the length.  When the high bit is set, the remaining bits are the length
 *  of length (either 1 byte (0x81) or 2 bytes (0x82) in practice).
 */
static uint16_t
ypDecodeLength(
    const uint8_t  *payload,
    uint16_t       *offset)
{
    uint16_t obj_len;

    obj_len = *(payload + *offset);
    if (obj_len == CERT_1BYTE) {
        (*offset)++;
        obj_len = *(payload + *offset);
    } else if (obj_len == CERT_2BYTE) {
        (*offset)++;
        obj_len = ntohs(*(uint16_t *)(payload + *offset));
        (*offset)++;
    }

    return obj_len;
}


/*
 *  Decodes the type of value in `payload` at `offset`, fills `tlv` with the
 *  type and returns the length.
 */
static uint16_t
ypDecodeTLV(
    yf_asn_tlv_t   *tlv,
    const uint8_t  *payload,
    uint16_t       *offset)
{
    uint8_t  val = *(payload + *offset);
    uint16_t len = 0;

    tlv->class = (val & 0xD0) >> 6;
    tlv->p_c = (val & 0x20) >> 5;
    tlv->tag = (val & 0x1F);

    (*offset)++;

    len = ypDecodeLength(payload, offset);
    (*offset)++;

    if (tlv->tag == CERT_NULL) {
        *offset += len;
        return ypDecodeTLV(tlv, payload, offset);
    }

    return len;
}


static gboolean
ypDecodeOID(
    const uint8_t  *payload,
    uint16_t       *offset,
    uint8_t         obj_len)
{
    uint32_t tobjid;

    if (obj_len == 9) {
        /* pkcs-9 */
        tobjid = ntohl(*(uint32_t *)(payload + *offset));
        if (tobjid != CERT_PKCS) {
            return FALSE;
        }
        *offset += 8;
    } else if (obj_len == 10) {
        /* LDAP Domain Component */
        tobjid = ntohl(*(uint32_t *)(payload + *offset));
        if (tobjid != CERT_DC) {
            return FALSE;
        }
        *offset += 9;
    } else if (obj_len == 3) {
        *offset += 2;
    } else {
        /* this isn't the usual id-at, pkcs, or dc - so lets ignore it */
        return FALSE;
    }

    return TRUE;
}


static uint8_t
ypGetSequenceCount(
    const uint8_t  *payload,
    uint16_t        seq_len)
{
    uint16_t     offsetptr = 0;
    uint16_t     len = 0;
    uint16_t     obj_len;
    uint8_t      count = 0;
    yf_asn_tlv_t tlv;

    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    while (tlv.tag == CERT_SET && len < seq_len) {
        len += obj_len + 2;
        count++;
        offsetptr += obj_len;
        obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    }

    return count;
}

static uint8_t
ypGetExtensionCount(
    const uint8_t  *payload,
    uint16_t        ext_len)
{
    uint16_t     offsetptr = 0;
    yf_asn_tlv_t tlv;
    uint16_t     len = 2;
    uint16_t     obj_len = 0;
    uint16_t     id_ce;
    uint8_t      obj_type = 0;
    uint8_t      count = 0;

    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    while (tlv.tag == CERT_SEQ && len < ext_len) {
        len += obj_len + 2;
        if (*(payload + offsetptr) == CERT_OID) {
            id_ce = ntohs(*(uint16_t *)(payload + offsetptr + 2));
            if (id_ce == CERT_IDCE) {
                obj_type = *(payload + offsetptr + 4);
                switch (obj_type) {
                  case 14:
                  /* subject key identifier */
                  case 15:
                  /* key usage */
                  case 16:
                  /* private key usage period */
                  case 17:
                  /* alternative name */
                  case 18:
                  /* alternative name */
                  case 29:
                  /* authority key identifier */
                  case 31:
                  /* CRL dist points */
                  case 32:
                  /* Cert Policy ID */
                  case 35:
                  /* Authority Key ID */
                  case 37:
                    count++;
                  default:
                    break;
                }
            }
        }
        offsetptr += obj_len;
        obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    }

    return count;
}


static gboolean
ypDecodeSSLCertificate(
    yfDPIContext_t   *ctx,
    yfSSLCertFlow_t **sslCert,
    const uint8_t    *payload,
    unsigned int      payloadSize,
    yfFlow_t         *flow,
    uint16_t          offsetptr)
{
    uint32_t         sub_cert_len;
    uint16_t         tot_ext_len = 0;
    uint16_t         ext_hold = 0;
    uint8_t          seq_count;
    uint8_t          obj_type = 0;
    yf_asn_tlv_t     tlv;
    yfSSLObjValue_t *sslObject = NULL;
    uint16_t         obj_len;
    uint16_t         set_len;
    uint16_t         off_hold;
    uint16_t         id_ce;

    /* we should start with the length of inner cert */
    if ((size_t)offsetptr + 5 > payloadSize) {
        return FALSE;
    }

    sub_cert_len = (ntohl(*(uint32_t *)(payload + offsetptr)) & 0xFFFFFF00) >>
        8;

    /* only continue if we have enough payload for the whole cert */
    if (offsetptr + sub_cert_len > payloadSize) {
        return FALSE;
    }

    offsetptr += 3;
    /* this is a CERT which is a sequence with length > 0x7F [0x30 0x82]*/

    (*sslCert)->hash.len = 0;

    if (ntohs(*(uint16_t *)(payload + offsetptr)) != 0x3082) {
        return FALSE;
    }

    /* 2 bytes for above, 2 for length of CERT */
    /* Next we have a signed CERT so 0x3082 + length */

    offsetptr += 8;

    /* A0 is for explicit tagging of Version Number */
    /* 03 is an Integer - 02 is length, 01 is for tagging */
    if (*(payload + offsetptr) == CERT_EXPLICIT) {
        offsetptr += 4;
        (*sslCert)->version = *(payload + offsetptr);
        offsetptr++;
    } else {
        /* default version is version 1 [0] */
        (*sslCert)->version = 0;
    }

    /* serial number */
    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    if (obj_len > sub_cert_len) {
        return FALSE;
    }
    if (tlv.tag == CERT_INT) {
        (*sslCert)->serial.buf = (uint8_t *)payload + offsetptr;
        (*sslCert)->serial.len = obj_len;
    }
    offsetptr += obj_len;

    /* signature */
    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    if (obj_len > sub_cert_len) {
        return FALSE;
    }

    if (tlv.tag != CERT_SEQ) {
        offsetptr += obj_len;
    } else {
        obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
        if (tlv.tag == CERT_OID) {
            if (obj_len > sub_cert_len) {
                return FALSE;
            }
            (*sslCert)->sig.buf = (uint8_t *)payload + offsetptr;
            (*sslCert)->sig.len = obj_len;
        }
        offsetptr += obj_len;
    }

    /* issuer - sequence */

    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    if (obj_len > sub_cert_len) {
        return FALSE;
    }

    if (tlv.tag == CERT_SEQ) {
        seq_count = ypGetSequenceCount((payload + offsetptr), obj_len);
    } else {
        return FALSE;
    }

    sslObject = (yfSSLObjValue_t *)fbSubTemplateListInit(
        &((*sslCert)->issuer), 3,
        YAF_SSL_SUBCERT_FLOW_TID, sslSubTemplate, seq_count);
    while (seq_count && sslObject) {
        set_len = ypDecodeTLV(&tlv, payload, &offsetptr);
        if (set_len >= sub_cert_len) {
            return FALSE;
        }
        if (tlv.tag != CERT_SET) {
            break;
        }
        off_hold = offsetptr;
        obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
        if (obj_len >= sub_cert_len) {
            return FALSE;
        }
        if (tlv.tag != CERT_SEQ) {
            break;
        }
        obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
        if (obj_len >= sub_cert_len) {
            return FALSE;
        }

        if (tlv.tag != CERT_OID) {
            break;
        }

        if (!ypDecodeOID(payload, &offsetptr, obj_len)) {
            sslObject++;
            seq_count--;
            offsetptr = off_hold + set_len;
            continue;
        }

        sslObject->obj_id = *(payload + offsetptr);
        offsetptr += 2;
        sslObject->obj_value.len = ypDecodeLength(payload, &offsetptr);
        if (sslObject->obj_value.len >= sub_cert_len) {
            sslObject->obj_value.len = 0;
            return FALSE;
        }
        offsetptr++;
        /* OBJ VALUE */
        sslObject->obj_value.buf = (uint8_t *)payload + offsetptr;
        offsetptr += sslObject->obj_value.len;
        seq_count--;
        sslObject++;
    }

    /* VALIDITY is a sequence of times */
    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    if (obj_len >= sub_cert_len) {
        return FALSE;
    }
    if (tlv.tag != CERT_SEQ) {
        return FALSE;
    }

    /* notBefore time */
    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    if (obj_len >= sub_cert_len) {
        return FALSE;
    }
    if (tlv.tag != CERT_TIME) {
        return FALSE;
    }
    (*sslCert)->not_before.buf = (uint8_t *)payload + offsetptr;
    (*sslCert)->not_before.len = obj_len;

    offsetptr += obj_len;

    /* not After time */
    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    if (obj_len >= sub_cert_len) {
        return FALSE;
    }
    if (tlv.tag != CERT_TIME) {
        return FALSE;
    }
    (*sslCert)->not_after.buf = (uint8_t *)payload + offsetptr;
    (*sslCert)->not_after.len = obj_len;

    offsetptr += obj_len;

    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    if (obj_len >= sub_cert_len) {
        return FALSE;
    }

    /* subject - sequence */
    if (tlv.tag == CERT_SEQ) {
        seq_count = ypGetSequenceCount((payload + offsetptr), obj_len);
    } else {
        return FALSE;
    }

    sslObject = (yfSSLObjValue_t *)fbSubTemplateListInit(
        &((*sslCert)->subject), 3,
        YAF_SSL_SUBCERT_FLOW_TID, sslSubTemplate, seq_count);

    while (seq_count && sslObject) {
        set_len = ypDecodeTLV(&tlv, payload, &offsetptr);
        if (set_len >= sub_cert_len) {
            return FALSE;
        }
        off_hold = offsetptr;
        if (tlv.tag != CERT_SET) {
            break;
        }
        obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
        if (obj_len >= sub_cert_len) {
            return FALSE;
        }

        if (tlv.tag != CERT_SEQ) {
            break;
        }
        obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
        if (obj_len >= sub_cert_len) {
            return FALSE;
        }
        if (tlv.tag != CERT_OID) {
            break;
        }

        if (!ypDecodeOID(payload, &offsetptr, obj_len)) {
            sslObject++;
            seq_count--;
            offsetptr = off_hold + set_len;
            continue;
        }

        sslObject->obj_id = *(payload + offsetptr);
        offsetptr += 2;
        sslObject->obj_value.len = ypDecodeLength(payload, &offsetptr);
        if (sslObject->obj_value.len >= sub_cert_len) {
            sslObject->obj_value.len = 0;
            return FALSE;
        }
        offsetptr++;
        /* OBJ VALUE */
        sslObject->obj_value.buf = (uint8_t *)payload + offsetptr;
        offsetptr += sslObject->obj_value.len;
        seq_count--;
        sslObject++;
    }

    /* subject public key info */
    /* this is a sequence of a sequence of algorithms and public key */
    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    if (obj_len >= sub_cert_len) {
        return FALSE;
    }
    /* this needs to be a sequence */
    if (tlv.tag != CERT_SEQ) {
        offsetptr += obj_len;
    } else {
        /* this is also a seq */
        obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
        if (obj_len >= sub_cert_len) {
            return FALSE;
        }
        if (tlv.tag != CERT_SEQ) {
            offsetptr += obj_len;
        } else {
            obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
            if (obj_len >= sub_cert_len) {
                return FALSE;
            }
            /* this is the algorithm id */
            if (tlv.tag == CERT_OID) {
                (*sslCert)->pkalg.buf = (uint8_t *)payload + offsetptr;
                (*sslCert)->pkalg.len = obj_len;
            }
            offsetptr += obj_len;
            obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
            if (obj_len >= sub_cert_len) {
                return FALSE;
            }
            /* this is the actual public key */
            if (tlv.tag == CERT_BITSTR) {
                (*sslCert)->pklen = obj_len;
            }
            offsetptr += obj_len;
        }
    }

    /* EXTENSIONS! - ONLY AVAILABLE FOR VERSION 3 */
    /* since it's optional - it has a tag if it's here */
    obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
    if (obj_len >= sub_cert_len) {
        return FALSE;
    }

    if ((tlv.class != 2) || ((*sslCert)->version != 2)) {
        /* no extensions */
        ext_hold = offsetptr;
        fbSubTemplateListInit(&((*sslCert)->extension), 3,
                              YAF_SSL_SUBCERT_FLOW_TID, sslSubTemplate, 0);
    } else {
        uint16_t ext_len;
        obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
        tot_ext_len = obj_len;
        if (obj_len >= sub_cert_len) {
            return FALSE;
        }

        ext_hold = offsetptr;

        if (tlv.tag == CERT_SEQ) {
            seq_count = ypGetExtensionCount((payload + offsetptr), obj_len);
        } else {
            return FALSE;
        }
        /* extensions */
        sslObject = (yfSSLObjValue_t *)fbSubTemplateListInit(
            &((*sslCert)->extension), 3,
            YAF_SSL_SUBCERT_FLOW_TID, sslSubTemplate, seq_count);
        /* exts is a sequence of a sequence of {id, critical flag, value} */
        while (seq_count && sslObject) {
            ext_len = ypDecodeTLV(&tlv, payload, &offsetptr);
            if (ext_len >= sub_cert_len) {
                return FALSE;
            }

            if (tlv.tag != CERT_SEQ) {
                return FALSE;
            }

            off_hold = offsetptr;
            obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
            if (obj_len >= ext_len) {
                return FALSE;
            }

            if (tlv.tag != CERT_OID) {
                return FALSE;
            }

            id_ce = ntohs(*(uint16_t *)(payload + offsetptr));
            if (id_ce != CERT_IDCE) {
                /* jump past this */
                offsetptr = off_hold + ext_len;
                continue;
            }
            offsetptr += 2;
            obj_type = *(payload + offsetptr);
            offsetptr++;
            obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
            if (obj_len >= ext_len) {
                return FALSE;
            }
            if (tlv.tag == CERT_BOOL) {
                /* this is optional CRITICAL flag */
                offsetptr += obj_len;
                obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
                if (obj_len >= ext_len) {
                    return FALSE;
                }
            }
            switch (obj_type) {
              case 14:
              /* subject key identifier */
              case 15:
              /* key usage */
              case 16:
              /* private key usage period */
              case 17:
              /* alternative name */
              case 18:
              /* alternative name */
              case 29:
              /* authority key identifier */
              case 31:
              /* CRL dist points */
              case 32:
              /* Cert Policy ID */
              case 35:
              /* Authority Key ID */
              case 37:
                /* ext. key usage */
                sslObject->obj_id = obj_type;
                sslObject->obj_value.len = obj_len;
                sslObject->obj_value.buf = (uint8_t *)payload + offsetptr;
                offsetptr += obj_len;
                seq_count--;
                sslObject++;
                break;
              default:
                offsetptr = off_hold + ext_len;
                continue;
            }
        }
    }

    if (cert_hash_export) {
        /* signature again */
        offsetptr = ext_hold + tot_ext_len;
        if (offsetptr > payloadSize) {
            return TRUE;
        }
        obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
        if (obj_len > sub_cert_len) {
            return TRUE;
        }

        if (tlv.tag == CERT_SEQ) {
            obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
            if (tlv.tag != CERT_OID) {
                return TRUE;
            }

            offsetptr += obj_len;
            if (offsetptr > payloadSize) {
                return TRUE;
            }
            obj_len = ypDecodeTLV(&tlv, payload, &offsetptr);
            /*get past padding */
            offsetptr++;
            if ((offsetptr + obj_len) > payloadSize) {
                return TRUE;
            }
            if (tlv.tag != CERT_BITSTR) {
                return TRUE;
            }
            if ((obj_len - 1) % 16) {
                return TRUE;
            }
            (*sslCert)->hash.len = obj_len - 1;
            (*sslCert)->hash.buf = (uint8_t *)payload + offsetptr;
        }
    }

    return TRUE;
}
#endif  /* YAF_ENABLE_DPI */


/**
 * ydpInitialize
 *
 * Processes the plugin's arguments to determine whether to enable (1)full
 * certificate export and (2)certifcate hash export.
 *
 */
int
ydpInitialize(
    int             argc,
    char           *argv[],
    uint16_t        applabel,
    gboolean        applabelOnly,
    GError        **err)
{
#if YAF_ENABLE_DPI
    if (!applabelOnly) {
        /* the first argument is the plugin's name */

        /* the second argument determines whether to enable export of the
         * complete binary X.509 certificate instead of DPI processing */
        if (argc > 1 && (0 == strcmp("true", argv[1]))) {
            full_cert_export = TRUE;
            ssl_dpi_off = TRUE;
            g_debug("SSL [Full] Certificate Export Enabled.");
        }

        /* the third argument determines whether to enable export of the
         * certificate's hash; if set, re-enables DPI export if
         * full_cert_export disabled it */
        if (argc > 2 && (0 == strcmp("true", argv[2]))) {
            cert_hash_export = TRUE;
            ssl_dpi_off = FALSE;
            g_debug("SSL Certificate Hash Export Enabled.");
        }
    }
#endif

    return 1;
}
